/* This program generates impulse responses of macro variables to updated Romer-Romer(2004) monetary policy shocks using Jorda (2005) local projections excluding privately held public debt*/

*** LP settings
{
    
	global sample "inrange(date,ym(1969,3),ym(2007,12))" // Wieland RR sample

	local lp_hor = 36
	local lp_lags = 12
	local shock_lags = 12
	local debtvar "privmrkval"
	local shock "resid_full"
	local yvars "lindpro lcpiaucsl unrate fedfunds"
	local xvars "lindpro lcpiaucsl lppiaco fedfunds debtnorm"	
	 
}

*** load data
{ 

	use "$projdir/dta/cln/data.dta", clear
	
	tsset date

	gen sample = $sample
	keep if sample==1

}

*** process data
{	
	
	gen hor = _n-1
	
	*generate normalized debt measure
	gen debt_gdp = l.`debtvar'/l.mngdp
	egen debtnorm = std(debt_gdp) if sample==1
	drop debt_gdp

	summ `shock'
	replace `shock' = `shock'/r(sd)

	gen lindpro = 100*log(indpro)
	gen lcpiaucsl = 100*log(cpiaucsl)
	gen lppiaco = 100*log(ppiaco)
	
	local graphs ""	
	
	replace `shock'=0 if `shock'==.

	 *demean x variables (required for Oaxaca-Blinder IRF) and create RHS
     foreach var in `xvars' `xvars_diff' `shock'{

		qui summ `var'
		replace  `var' = `var'-r(mean)

     }

}

*** estimate IRFs
foreach var in `yvars'{
	
	if "`var'"=="lindpro"{
		local title "Industrial Production"
	}
	if "`var'"=="lcpiaucsl"{
		local title "Consumer Price Index"
	}
	if "`var'"=="unrate"{
		local title "Unemployment Rate"
	}
	if "`var'"=="fedfunds"{
		local title "Federal Funds Rate"
	}
	if "`var'"=="gs1"{
		local title "One-Year Treasury"
	}

	*initialize IRFs
	gen irf_`var' = .
	gen upper_`var' = .
	gen lower_`var' = .

	local start = 0
	
	forvalues i = 0/`lp_hor'{
		
		*Generate LHS variable
		gen ydiff_h`i' = F`i'.`var' - L.`var'
		
		*Run regressions
		ivreg2 ydiff_h`i' L(1/`lp_lags')(`xvars') L(0/`shock_lags')(`shock'), robust bw(auto)
		
		drop ydiff*
		
		replace irf_`var' = _b[`shock'] if hor==`i'
		replace lower_`var' = _b[`shock']-_se[`shock'] if hor==`i'
		replace upper_`var' = _b[`shock']+_se[`shock'] if hor==`i'
		
	}
	
	* plot IRF
	tw ///
	(rarea upper_`var' lower_`var' hor, bcolor(gs14%75) lw(none)) ///
	(line irf_`var' hor, clc(black) c(l ) clp(l ) ms(i)) ///
	if hor<=`lp_hor' ///
	, nodraw name(irf_mp_`var', replace) yline(0) xlabel(0(6)36) legend(off) title("`title'") ytitle("Percent") xtitle("Months")

	local newgraph "irf_mp_`var'"
	local graphs "`graphs' `newgraph'"	
		
}

***combine IRFs
{

	graph combine `graphs', name(irf_rrshock, replace) rows(2) 
	graph export "$projdir/out/irf_rrshock.pdf", replace

}