/*

Compilation_EstSample.do

Note 1: Update the estimation sample using the updated iMaPP database, 
        by dropping iMaPP-related variables from quarterly_wp_macropru_final4.dta,
	    re-constructing them, and saving the data as quarterly_wp_macropru_final5.dta.
Note 2: Use the iMaPP_Q.dta, generated by Compilation_iMaPP.do.  
Note 3: Merge data of household credit breakdown and extended household credit 
        from "procesed_data_v2.dta". 
	  
Last Updated: August 17, 2022
Previous version: April 23, 2022
(C:\Users\MNarita\OneDrive - International Monetary Fund (PRD)\MachiKuma\iMaPP paper\Analysis\Codes\Archive)

*/

*===============================================================================
* Step 0: Set up
*===============================================================================
clear all 
set maxvar 10000
set more off
set excelxlsxlargefile on
pause off
capture log close	   

// Set the log file
local today : display %tdCYND date(c(current_date), "DMY")  // Date (YYYYMMDD)
log using "../Results/Macropru/Compilation_EstSample_`today'.txt", text replace

// Control flag variables
local str_version 8_iMaPP2021
local update_iMaPP = 1	// If 1, update iMaPP using iMaPP_Q.dta in the Data foloder
local extended_data_filepath "../Codes/data_processing_v2/processed_data_v2.dta"
local extend_beyond2016 = 0	// If 1, the dataset will include data beyond 2016Q4

*===============================================================================
* Step 1: Load the data used for WP with additional data from Adrian (v61)
*===============================================================================
// Load the current data file (quarterly_wp_macropru_final#.dta)
use  "../Data/quarterly_wp_macropru_final61.dta", clear
xtset isocode dateq
sort  isocode dateq
if `extend_beyond2016' == 1 tsappend, last(2021q4) tsfmt(tq) 
rename iso3 iso3_org 
decode isocode, gen(iso3)
rename ifscode ifscode_org
egen ifscode = mean(ifscode_org), by(isocode)
isid ifscode dateq
xtset ifscode dateq
sort  ifscode dateq
*xtset ifscode quarter
*sort  ifscode quarter
*rename quarter dateq

if `extend_beyond2016' == 1 {
	local extend_varlist AE 
	foreach v of local extend_varlist {
		rename `v' `v'_org
		egen `v' = mean(`v'_org), by(ifscode)
		drop `v'_org
	}	
}


// Drop un-necessary variables
capture drop  rr_nrd 
capture drop  _merge
capture drop  diff*
capture drop dltv_mean 
capture drop dltv_min 
capture drop dltv_max

capture drop iso3_org ifscode_org
order iso3 isocode dateq ifscode

*===============================================================================
* Step 2: Merge additoinal data on household credit breakdown
*===============================================================================
local x_varlist x_bankloan lab_bankloan x_bankmort lab_bankmort
if `extend_beyond2016' == 1 local x_varlist `x_varlist' x_hhdebt lab_hhdebt PCPI_IX_Q
merge 1:1 ifscode dateq using "`extended_data_filepath'", nogenerate keepusing(`x_varlist') 

rename cpi cpi_org 
gen cpi = cpi_org
replace cpi = . if iso3 == "DEU" & dateq <= yq(1990,4)	// Germany before the merger
if `extend_beyond2016' == 1 replace cpi = PCPI_IX_Q

drop if missing(iso3) 

// Create further breakdown of household credit
qui gen x_banknonmort = x_bankloan - x_bankmort
label variable x_banknonmort "Non-mortgage loans to households by banks"

qui gen x_nonbankloan = hhdebt - x_bankloan
label variable x_nonbankloan "Loans to households by nonbanks"
// Remove x_nonbankloan if it is less than one percent of hhdebt 
replace x_nonbankloan = . if x_nonbankloan < (hhdebt*0.01)

qui gen x_hhdebtexbm = hhdebt - x_bankmort
label variable x_hhdebtexbm "Household credit excluding mortgage loans by banks"

sort ifscode dateq
local new_varlist bankloan bankmort banknonmort nonbankloan hhdebtexbm
if `extend_beyond2016' == 1 local new_varlist `new_varlist' hhdebt
// A prefix "x_" to variable names (e.g., x_lnrbankloan, d4x_lnrbankloan) is added to make these variables listed after the original data series. 
foreach v of local new_varlist {
    qui gen x_lnr`v' = ln(x_`v'/cpi) 
	local lab_tmp : variable label x_`v'
	label variable x_lnr`v' " log real `lab_tmp'"
    qui gen d4x_lnr`v' = 100*(x_lnr`v' - L4.x_lnr`v')
	label variable d4x_lnr`v' "d4 100 log real `lab_tmp'"
	
	// Winsorize
	winsor d4x_lnr`v', gen(d4x_lnr`v'_w1) p(0.01)	
}
if `extend_beyond2016' == 0 {
	drop cpi
	rename cpi_org cpi
}


*===============================================================================
* Step 3: Update iMaPP variables
*===============================================================================			  
if `update_iMaPP' {
	*===============================================================================
	* Step 3-1: Read an updated iMaPP data file
	*===============================================================================
	// Back-up the old variables
	drop *_old
	local varlist ccb conservation capital capital_gen capital_hh capital_corp capital_fx ///
			  lvr llp lcg lcg_gen lcg_hh lcg_corp loanr loanr_hh loanr_corp lfc /// 
			  ltv dsti tax lr ltd lfx rr rr_fcd sifi ot sum_17 ///
			  ltv_level dltv_level dltv_level_w dltv_level_w1 dltv_level_w1_tight dltv_level_w1_loose ///
			  dltv_level_tight dltv_level_tight10 dltv_level_tight20 dltv_level_tight30 ///
			  dltv_level_tight10plus dltv_level_tight1025 dltv_level_tight025 

    // Save old `v1' as `v1'_old
	foreach v1 in `varlist' {
		capture rename `v1' `v1'_old
	}
	label var lr_old "liquidity requirement (old)"
	foreach v1 of varlist LTV_* {
		capture rename `v1' `v1'_old
	}

	// Merge the updated iMaPP_Q.dta
	merge 1:1 iso3 dateq using "../Data/iMaPP_Q.dta"
	*merge 1:1 iso3 dateq using "\\data1\MCM\DRAFTS\MP\iMaPP Database\2 iMaPP - Master\2019 update\3 Release\1 Files\iMaPP_Q.dta"
	*merge 1:1 iso3 dateq using "\\data1\MCM\DRAFTS\MP\iMaPP Database\2 iMaPP - Master\2020 update\3 Release\1 Files\iMaPP_Q.dta"
	*rename dateq quarter
	drop if _merge == 2  // Countries outside the estimation sample
	*sort  ifscode quarter
	sort  ifscode dateq

	// Rename the new variables
	/*
	local varlist_new CCB Conservation ///
				  Capital Capital_Gen Capital_HH Capital_Corp Capital_FX ///
				  LVR LLP LCG LCG_Gen LCG_HH LCG_Corp LoanR LoanR_HH LoanR_Corp LFC ///
				  LTV DSTI Tax Liquidity LTD LFX RR RR_FCD SIFI OT SUM_17	
	*/
	rename CCB ccb
	rename Conservation conservation
	rename Capital capital
	rename Capital_Gen capital_gen	
	rename Capital_HH capital_hh 
	rename Capital_Corp capital_corp 
	rename Capital_FX capital_fx 					   
	rename LVR lvr
	rename LLP llp 
	rename LCG lcg 
	rename LCG_Gen lcg_gen 
	rename LCG_HH lcg_hh
	rename LCG_Corp lcg_corp
	rename LoanR loanr
	rename LoanR_HH loanr_hh
	rename LoanR_Corp loanr_corp
	rename LFC lfc
	rename LTV ltv 
	rename DSTI dsti 
	rename Tax tax 
	rename Liquidity lr 
	rename LTD ltd 
	rename LFX lfx 
	rename RR rr 
	rename RR_FCD rr_fcd 
	rename SIFI sifi 
	rename OT ot 
	rename SUM_17 sum_17

	// LTV_average variables
	// [August 25, 2020] For comparion only. Be careful for any confusion with the correct variables to use, which are reconstructed in the estimation code.

	* local varlist dltv_level_tight dltv_level_tight10 dltv_level_tight1025
	gen 	  ltv_level  = LTV_Qmax
	gen       dltv_level = d.LTV_Qmax
	label var ltv_level  "LTV_Qmax. LTV_Qmax is the Quarterly Max of LTV_average"
	label var dltv_level "d.LTV_Qmax. LTV_Qmax is the Quarterly Max of LTV_average"

	gen dltv_level_tight=dltv_level   if dltv_level<=0
	gen dltv_level_tight10=dltv_level if dltv_level<=0 & dltv_level>-10
	gen dltv_level_tight20=dltv_level if dltv_level<=-10 & dltv_level>-20
	gen dltv_level_tight30=dltv_level if dltv_level<=-20
	replace dltv_level_tight10=0 if missing(dltv_level_tight10) & !missing(dltv_level)
	replace dltv_level_tight20=0 if missing(dltv_level_tight20) & !missing(dltv_level)
	replace dltv_level_tight30=0 if missing(dltv_level_tight30) & !missing(dltv_level)
	label var dltv_level_tight   "dltv_level: tightening only"
	label var dltv_level_tight10 "dltv_level: tightening by less than 10 ppt only"
	label var dltv_level_tight20 "dltv_level: tightening by (-20, -10] ppt only"
	label var dltv_level_tight30 "dltv_level: tightening by more than 20 ppt only"

	gen dltv_level_tight10plus=dltv_level if dltv_level<=-10
	replace dltv_level_tight10plus=0 if missing(dltv_level_tight10plus) & !missing(dltv_level)
	label var dltv_level_tight10plus "dltv_level: tightening by 10 ppt or less only"

	gen dltv_level_tight1025=dltv_level if dltv_level<=-10 & dltv_level>-25
	replace dltv_level_tight1025=0 if missing(dltv_level_tight1025) & !missing(dltv_level)
	label var dltv_level_tight1025 "dltv_level: tightening by (-25, -10] ppt only"

	gen dltv_level_tight025=dltv_level if dltv_level<0 & dltv_level>-25
	replace dltv_level_tight025=0 if missing(dltv_level_tight025) & !missing(dltv_level)
	label var dltv_level_tight025 "dltv_level: tightening by less than -25 ppt only"

	// Difference between new and old variables
	local varlist ccb conservation capital lvr llp lcg loanr lfc /// 
				  ltv dsti tax lr ltd lfx rr sifi ot sum_17 ///
				  dltv_level dltv_level_tight dltv_level_tight10 dltv_level_tight1025
	local cnt = 0
	foreach var1 in `varlist' {
		local cnt = `cnt'+1
		gen diff_`var1'      = `var1' - `var1'_old
		di "========================================"
		di "`cnt' `var1': Number of policy actions (old)"
		count if `var1'_old !=0
		di "`cnt' `var1': Number of policy actions (new)"
		count if `var1' !=0	
		di "`cnt' diff_`var1': Number of changes (new vs. old)"
		count if diff_`var1' !=0
	}

	*===============================================================================
	* Step 3-2: Re-generate MaPP-related variables
	*===============================================================================
	*****Generate Y-o-Y variables*****
	local varlist ccb conservation capital lvr llp lcg loanr lfc /// 
				  ltv dsti tax lr ltd lfx rr sifi ot sum_17
	foreach var1 in `varlist' {
	 // Drop the old variable
	 *capture drop `var1'_yoy  
	 capture rename `var1'_yoy `var1'_yoy_old
	 // Generate the variable with new data
	 bysort ifscode: gen `var1'_yoy=(`var1'[_n]+`var1'[_n-1]+`var1'[_n-2]+`var1'[_n-3])/4 
	 label var `var1'_yoy "`var1': Average from t-3 to t" 
	 *label var `var1'_yoy "`var1'" 
	}

	*****Generate iMaPP indices*****
	// Added ltd_yoy and ot_yoy to allsum
	*gen allsum=dsti+ltv+rr+lvr+lcg+lfx+lfc+llp+capital+tax+conservation+sifi+ccb+lr+loanr
	*gen allsum_yoy=dsti_yoy+ltv_yoy+rr_yoy+lvr_yoy+tax_yoy+lcg_yoy+lfx_yoy+lfc_yoy+llp_yoy+capital_yoy+conservation_yoy+sifi_yoy+ccb_yoy+lr_yoy+loanr_yoy
	// Drop the old variable
	*gen lallsum_yoy_old = lallsum_yoy
	capture drop allsum 
	rename allsum_yoy  allsum_yoy_old
	rename lallsum_yoy lallsum_yoy_old
	// Re-generate the variable
	gen allsum=dsti+ltv+rr+lvr+lcg+lfx+lfc+llp+capital+tax+conservation+sifi+ccb+lr+loanr+ltd+ot
	gen allsum_yoy=dsti_yoy+ltv_yoy+rr_yoy+lvr_yoy+tax_yoy+lcg_yoy+lfx_yoy+lfc_yoy+llp_yoy+capital_yoy+conservation_yoy+sifi_yoy+ccb_yoy+lr_yoy+loanr_yoy+ltd_yoy+ot_yoy
	gen lallsum_yoy = L.allsum_yoy
	label var allsum_yoy  "Sum of all 17 measures"
	*label var lallsum_yoy "Lag of the sum of all measures but ot (other)"
	label var allsum_yoy "MaPP (All, Sum)"
	*br iso3 quarter allsum_yoy lallsum_yoy lallsum_yoy_old

	*gen all7sum=dsti+ltv+loanr+lcg+llp+rr+ccb
	*gen all7sum_yoy=dsti_yoy+ltv_yoy+loanr_yoy+lcg_yoy+llp_yoy+rr_yoy+ccb_yoy
	// Drop the old variable
	capture drop allloansum 
	rename allloansum_yoy  allloansum_yoy_old
	rename lallloansum_yoy lallloansum_yoy_old
	// Re-generate the variable
	gen allloansum=dsti+ltv+lcg+llp+loanr+ltd+lfc
	gen allloansum_yoy=dsti_yoy+ltv_yoy+lcg_yoy+llp_yoy+loanr_yoy+ltd_yoy+lfc_yoy
	gen lallloansum_yoy = L.allloansum_yoy
	*label var allloansum_yoy  "Sum of loan-targeted measures"
	*label var lallloansum_yoy "Lag of the sum of loan-targeted measures"
	label var allloansum_yoy "MaPP (Loan, Sum)"
	label var lallloansum_yoy "MaPP (Loan, Sum)"

	// Drop the old variable
	capture drop demand_side 
	rename demand_side_yoy  demand_side_yoy_old
	rename ldemand_side_yoy ldemand_side_yoy_old
	// Re-generate the variable
	gen demand_side=dsti+ltv
	gen demand_side_yoy=dsti_yoy+ltv_yoy
	gen ldemand_side_yoy = L.demand_side_yoy
	*label var demand_side_yoy  "Sum of demand-side measures"
	*label var ldemand_side_yoy "Lag of the sum of demand-sid measures"
	label var demand_side_yoy "MaPP (Demand, Sum)"
	label var ldemand_side_yoy "MaPP (Demand, Sum)"

	// Drop the old variable
	capture drop supply_side_all 
	rename supply_side_all_yoy  supply_side_all_yoy_old
	rename lsupply_side_all_yoy lsupply_side_all_yoy_old
	// Re-generate the variable
	gen supply_side_all=rr+lvr+lcg+lfx+lfc+llp+capital+conservation+sifi+ccb+lr+loanr
	gen supply_side_all_yoy=rr_yoy+lvr_yoy+lcg_yoy+lfx_yoy+lfc_yoy+llp_yoy+capital_yoy+conservation_yoy+sifi_yoy+ccb_yoy+lr_yoy
	gen lsupply_side_all_yoy = L.supply_side_all_yoy
	*label var supply_side_all_yoy  "Sum of supply-side measures (all)"
	*label var lsupply_side_all_yoy "Lag of the sum of supply-sid measures (all)"
	label var supply_side_all_yoy "MaPP (Supply All, Sum)"
	label var lsupply_side_all_yoy "MaPP (Supply All, Sum)"

	// Drop the old variable
	capture drop supply_general 
	rename supply_general_yoy  supply_general_yoy_old
	rename lsupply_general_yoy lsupply_general_yoy_old
	// Re-generate the variable
	gen supply_general=rr+lr+lfx
	gen supply_general_yoy=rr_yoy+lr_yoy+lfx_yoy
	gen lsupply_general_yoy = L.supply_general_yoy
	*label var supply_general_yoy  "Sum of supply-side measures (general)"
	*label var lsupply_general_yoy "Lag of the sum of supply-sid measures (general)"
	label var supply_general_yoy "MaPP (Supply General, Sum)"
	label var lsupply_general_yoy "MaPP (Supply General, Sum)"

	// Drop the old variable
	capture drop supply_capital 
	rename supply_capital_yoy  supply_capital_yoy_old
	rename lsupply_capital_yoy lsupply_capital_yoy_old
	// Re-generate the variable
	gen supply_capital=lvr+ccb+conservation+capital
	gen supply_capital_yoy=lvr_yoy+ccb_yoy+conservation_yoy+capital_yoy
	gen lsupply_capital_yoy = L.supply_capital_yoy
	*label var supply_capital_yoy  "Sum of supply-side measures (capital)"
	*label var lsupply_capital_yoy "Lag of the sum of supply-sid measures (capital)"
	label var supply_capital_yoy "MaPP (Supply Capital, Sum)"
	label var lsupply_capital_yoy "MaPP (Supply Capital, Sum)"

	// Drop the old variable
	capture drop supply_loans 
	rename supply_loans_yoy  supply_loans_yoy_old
	rename lsupply_loans_yoy lsupply_loans_yoy_old
	// Re-generate the variable
	gen supply_loans=lcg+llp+loanr+ltd+lfc
	gen supply_loans_yoy=lcg_yoy+llp_yoy+loanr_yoy+ltd_yoy+lfc_yoy
	gen lsupply_loans_yoy = L.supply_loans_yoy
	*label var supply_loans_yoy  "Sum of supply-side measures (loan)"
	*label var lsupply_loans_yoy "Lag of the sum of supply-sid measures (loan)"
	label var supply_loans_yoy "MaPP (Supply Loans, Sum)"
	label var lsupply_loans_yoy "MaPP (Supply Loans, Sum)"

	local varlist dsti ltv loanr lcg lfc lfx llp lr ltd lvr capital conservation rr sifi tax ccb 
	foreach var1 in `varlist' {
		// Drop the old variable
		capture drop allsum_wo_`var1' allsum_wo_`var1'_yoy
		// Re-generate the variable
		*gen allsum_wo_`var1'=dsti+rr+lvr+lcg+lfx+lfc+llp+capital+tax+conservation+sifi+ccb+lr+loanr+ltv - `var1'
		*gen allsum_wo_`var1'_yoy=dsti_yoy+rr_yoy+lvr_yoy+tax_yoy+lcg_yoy+lfx_yoy+lfc_yoy+llp_yoy+capital_yoy+conservation_yoy+sifi_yoy+ccb_yoy+lr_yoy+loanr_yoy+ltv_yoy - `var1'_yoy
		gen allsum_wo_`var1'=dsti+rr+lvr+lcg+lfx+lfc+llp+capital+tax+conservation+sifi+ccb+lr+loanr+ltv+ltd - `var1'			
		gen allsum_wo_`var1'_yoy=dsti_yoy+rr_yoy+lvr_yoy+tax_yoy+lcg_yoy+lfx_yoy+lfc_yoy+llp_yoy+capital_yoy+conservation_yoy+sifi_yoy+ccb_yoy+lr_yoy+loanr_yoy+ltv_yoy+ltd_yoy - `var1'_yoy
		label var allsum_wo_`var1'_yoy "Sum of all measures but `var1' and ot: average from t-3 to t"
	}

	// Check
	*br iso3 quarter lcg lcg_old lcg_yoy lcg_yoy_old if lcg != lcg_old

	*===============================================================================
	* Step 3-3: Statistics
	*===============================================================================
	local list_var allsum_yoy allloansum_yoy demand_side_yoy supply_side_all_yoy ///
				   supply_loans_yoy supply_general_yoy supply_capital_yoy /// 
				   ccb_yoy conservation_yoy capital_yoy lvr_yoy llp_yoy lcg_yoy ///
				   loanr_yoy lfc_yoy ltv_yoy dsti_yoy tax_yoy lr_yoy ltd_yoy lfx_yoy ///
				   rr_yoy sifi_yoy ot_yoy dltv_level dltv_level_tight dltv_level_tight10 dltv_level_tight1025
	// Drop "old" variables for comparison
	local list_var_old
	foreach v1 in `list_var' {
		local list_var_old `list_var_old' `v1'_old
	}

				   
	// Whole sample
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate `list_var'     if SAMPLE_ALL_HHD ==1, stat(mean sd) column(stats)
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate `list_var_old' if SAMPLE_ALL_HHD ==1, stat(mean sd) column(stats)
	// AE sample
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate `list_var'     if SAMPLE_AE_HHD ==1, stat(mean sd) column(stats)
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate `list_var_old' if SAMPLE_AE_HHD ==1, stat(mean sd) column(stats)	
	// EMDE sample	
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate `list_var'	   if SAMPLE_EM_HHD ==1, stat(mean sd) column(stats)	
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate `list_var_old' if SAMPLE_EM_HHD ==1, stat(mean sd) column(stats)		

	// Drop "old" variables for comparison
	*drop `list_var_old'
}

*===============================================================================
* Step 4: Save
*===============================================================================
// Save
*sort ifscode quarter
sort ifscode dateq
save "../Data/quarterly_wp_macropru_final`str_version'.dta", replace

***********  END *************
log close
exit
