/*

Descriptive.do

Note: Use iMaPP_M.dta, which is produced by Compilation.do. 

Last Updated: October 28, 2023
Previous version: August 17, 2022
*/

*===============================================================================
* Step 0: Set up
*===============================================================================
clear all 
set maxvar 10000
set more off
set excelxlsxlargefile on
pause off
capture log close	  

// Set the log file
local today : display %tdCYND date(c(current_date), "DMY")  // Date (YYYYMMDD)
log using "../Results/Macropru/Descriptive_`today'.txt", text replace

// Set up
local SmplPeriod  "year >= 2000 & year <= 2016"
local flg_iMaPP_plus 0	// 1 if you want to compile (iMaPP_plus.dta) with the cumulative sum measures.
local flg_Fig_123 1		// 1 if you want to export (iMaPP_Fig1234.xlsx) for Fig 1-3
local flg_Fig_4	  1		// 1 if you want to export (Fig4_`today'.xlsx) for Fig 4
local flg_Fig_5   1		// 1 if you want to export (LTV_`today'.xlsx) for Fig 5
local flg_Fig_Stats 1	// 1 if you want to generate the summary stats (Appendix IV Table 3)
local flg_LTV_intro 1	// 1 if you want to generate the summary stats of LTV average, distinguishing introduction cases.

// Load iMaPP data (both dummy-type and average LTV data)
*use "\\data1\MCM\DRAFTS\MP\iMaPP Database\2 iMaPP - Master\2020 update\3 Release\1 Files\iMaPP_M.dta"
use "../Data/iMaPP_M.dta", clear
drop if ifscode == . 
drop if AE      == .  // Curacao, whose AE/EMDE are not available
xtset ifscode datem

*===============================================================================
* Step 1: Lists of MaPP indicators
*===============================================================================
levelsof iso3, local(Nat_Code)

local MaPP_17_list CCB Conservation LVR LLP Capital LTV DSTI ///
	               LCG LoanR LFC Tax Liquidity LTD LFX RR SIFI OT
local MaPP_16_list CCB Conservation LVR LLP Capital LTV DSTI ///
	               LCG LoanR LFC Tax Liquidity LTD LFX  SIFI OT
	  
local MaPP_17_list_T 
local MaPP_17_list_L
foreach x of local MaPP_17_list {
	local MaPP_17_list_T  `MaPP_17_list_T' `x'_T
	local MaPP_17_list_L  `MaPP_17_list_L' `x'_L
}

local MaPP_sub_list Capital_Gen Capital_HH Capital_Corp Capital_FX ///
	               RR_FCD RR_NRD LCG_Gen LCG_HH LCG_Corp LoanR_HH LoanR_Corp

*===============================================================================
* Step 2: Cumulative sum of MaPP indicators
*===============================================================================
if `flg_iMaPP_plus' == 1 {
	local flg_firsttime = 1
	foreach x of local MaPP_17_list {
		preserve 
		
		*levelsof iso3, local(Nat_Code)
		*local x LTV
		di " Cusum of `x' ======================================"
		// Reshape to wide.
		keep iso3 datem `x'  `x'_T  `x'_L
		reshape wide    `x'  `x'_T  `x'_L, i(datem) j(iso3) string
		
		// Running sum:  the sum of the first through jth observations on x
		local Cusum_list_A   
		local Cusum_list_T  
		local Cusum_list_L 
		local AE_list
		foreach k of local Nat_Code {
			gen `x'_cs_A`k' = sum(`x'`k')
			gen `x'_cs_T`k' = sum(`x'_T`k')
			gen `x'_cs_L`k' = sum(`x'_L`k')
			local Cusum_list_A `Cusum_list_A' `x'_cs_A`k' 
			local Cusum_list_T `Cusum_list_T' `x'_cs_T`k' 
			local Cusum_list_L `Cusum_list_L' `x'_cs_L`k'
		} 
		
		// Reshape back to long.
		keep datem `Cusum_list_A' `Cusum_list_T' `Cusum_list_L' 	 
		reshape long `x'_cs_A `x'_cs_T `x'_cs_L, i(datem) j(iso3) string
		rename `x'_cs_A  `x'_cs
		gen    `x'_cs_act   = `x'_cs_T + `x'_cs_L
		label var `x'_cs       "Cumulative sum: `x'"
		label var `x'_cs_T     "Cumulative sum: `x'_T"
		label var `x'_cs_L     "Cumulative sum: `x'_L"	
		label var `x'_cs_act   "Cumulative # of actions: `x'_cs_T + `x'_cs_L"	
		
		// Merge and save	
		if `flg_firsttime' == 1 {
			sort iso3 datem
			merge 1:1 iso3 datem using "../Data/iMaPP_M.dta", nogenerate
			save "../Data/iMaPP_Plus.dta", replace
			local flg_firsttime = 0
		} 
		else {
			sort iso3 datem
			merge 1:1 iso3 datem using "../Data/iMaPP_Plus.dta", nogenerate
			save "../Data/iMaPP_Plus.dta", replace
		}		
		
		restore	
	}
}

*===============================================================================
* Step 3.A: Prevalence and Frequency by instruments (Figure 1-3)
*===============================================================================
if `flg_Fig_123' == 1 {
	// Load dataset produced in  Step 2
	*use "../Data/iMaPP_Plus.dta", clear
	use "./iMaPP_Plus.dta", clear
	drop if ifscode == . 
	drop if AE      == .  // Curacao, whose AE/EMDE are not available	
	sort  ifscode datem
	xtset ifscode datem
	
	// for debug
	local MaPP_17_list CCB Conservation LVR LLP Capital LTV DSTI ///
					   LCG LoanR LFC Tax Liquidity LTD LFX RR SIFI OT
	local MaPP_16_list CCB Conservation LVR LLP Capital LTV DSTI ///
					   LCG LoanR LFC Tax Liquidity LTD LFX  SIFI OT	
	
	// Set frequency
	local frequency "A"  // A: Annual, Q: Quarterly, M: Monthly
				
	*------------------------------------------------------
	* 3.1 Use dummy (Fig 1-2)
	*------------------------------------------------------
	gen Sum_17_use = 0
	local use_list
	foreach x of local MaPP_17_list {
		gen       `x'_use     = (`x'_cs_act > 0)
		replace   Sum_17_use  = Sum_17_use + `x'_use
		label var `x'_use     "1 if ever used (`x'_cs_act > 0)"
		label var Sum_17_use  "sum of _use for all 17 instruments"	
		local use_list   `use_list' `x'_use
	}
	gen Sum_16_use = 0
	foreach x of local MaPP_16_list {
		replace  Sum_16_use = Sum_16_use + `x'_use
		label var Sum_16_use  "sum of _use for all (except for RR)"	
	}
	// Flag for the use of any instruments
	gen MaPP17_use = (Sum_17_use > 0)
	gen MaPP16_use = (Sum_16_use > 0)
	label var MaPP17_use  "1 if ever used any of the 17 MaPP instruments"	
	label var MaPP16_use  "1 if ever used any MaPP (except for RR)"	
	
	// Stats for All
	preserve 
		local sheet = "ALL"
		gen N   = AE + EMDE

		// First, collapse by datem to get the group sum at a monthly frequency
		collapse (sum) `use_list'  N MaPP16_use MaPP17_use (mean) Year dateq, by(datem)
		if "`frequency'" == "A" {
			// Then, collapse by Year to get annual series
			collapse (mean) N (lastnm) MaPP16_use MaPP17_use `use_list', by(Year)
			order N MaPP16_use MaPP17_use `use_list' 
			local time = "Year"
		}
		if "`frequency'" == "Q" {
			// Then, collapse by Year to get quarterly series
			collapse (mean) N (lastnm) MaPP16_use MaPP17_use `use_list', by(dateq)
			order N MaPP16_use MaPP17_use `use_list' 
			local time = "dateq"
		}		
		
		// Export to an Excel file
		export excel `time' N  *_use ///
			   using "../Data/iMaPP_Fig1234.xlsx", sheet("`sheet'_`frequency'") firstrow(variables) missing("NA") sheetreplace
	restore

	//  Stats for AE and EMDE
	preserve 
		gen N   = AE + EMDE

		// First, collapse by datem and AE to get the group sum at a monthly frequency
		collapse (sum) `use_list'  N MaPP16_use MaPP17_use (mean) Year dateq, by(datem AE)
		if "`frequency'" == "A" {
			// Then, collapse by Year to get annual series
			collapse (mean) N (lastnm) MaPP16_use MaPP17_use `use_list' , by(Year AE)
			order N MaPP16_use MaPP17_use `use_list' 
			local time = "Year"
		}
		if "`frequency'" == "Q" {
			// Then, collapse by Year to get quarterly series
			collapse (mean) N (lastnm) MaPP16_use MaPP17_use `use_list' , by(dateq AE)
			order N MaPP16_use MaPP17_use `use_list' 
			local time = "dateq"
		}		
			
		// Reshape to wide.
		reshape wide N MaPP16_use MaPP17_use `use_list' , i("`time'") j(AE)

		// Export to an Excel file
		forvalues group = 0/1 {
			if `group' == 0 {
				local sheet = "EMDE"
			}
			if `group' == 1 {
				local sheet = "AE"
			}
			export excel `time' N`group'  *_use`group'  ///
				   using "../Data/iMaPP_Fig1234.xlsx", sheet("`sheet'_`frequency'") firstrow(variables) missing("NA") sheetreplace
		}	
	restore
/*		
	*----------------------------------------------------------
	* 3.2 Frequency (# of actions per period: 2010M1-; Fig3)
	*----------------------------------------------------------
	// Number of actions for each instrument
	local list `MaPP_17_list'
	local act_list
	foreach x of local list {	
		gen       `x'_act   = `x'_T + `x'_L
		label var `x'_act   "# of actions: `x'_T + `x'_L"	
		local act_list   `act_list' `x'_act
	}
	gen     SUM_16_act = SUM_16_T + SUM_16_L
	replace SUM_16_act = (SUM_16_act > 0)
	label var SUM_16_act  "1 if any use of 16 instruments (except for RR)"	
	gen     SUM_17_act = SUM_17_T + SUM_17_L
	replace SUM_17_act = (SUM_17_act > 0)
	label var SUM_17_act  "1 if any use of 17 instruments
	
	// Total number of policy actions
	gen       N_actions_SUM_17 = SUM_17_T + SUM_17_L
	label var N_actions_SUM_17  "# of actions of all 17 categories (x: `x'_T + `x'_L)"	
	su N_actions_SUM_17
	local total_N_actions_SUM_17 = r(sum)
	di "# of actions of all 17 categories = `total_N_actions_SUM_17'"
	
	// List of the number-of-action variables
	local act_list   `act_list' SUM_16_act SUM_17_act	
	
	// Stats for All
	preserve
		local sheet = "ALL"
		
		// Focus on the recent years when many instruments have been introduced.
		keep if datem >= tm(2010M1) 
				
		// First, collapse by country to get the country-specific average #s of actions per month
		collapse (mean) `act_list', by(iso3)
		gen N = 1
		// Second, collapse to get the average # of actions per month
		collapse (sum) N (mean) `act_list'
		foreach x of local act_list {
			// Average number of actions per year
			replace `x' = `x'*12
		}
		
		// Export to an Excel file
		export excel N `act_list' ///
			   using "../Data/iMaPP_Fig1234.xlsx", sheet("Act`_`sheet'") firstrow(variables) missing("NA") sheetreplace
	restore

	//  Stats for AE and EMDE
	preserve 
		local sheet = "AEEM"
		
		// Focus on the recent years when many instruments have been introduced.
		keep if datem >= tm(2010M1)	
		
		// First, collapse by country to get the country-specific average #s of actions per month
		collapse (lastnm) AE (mean) `act_list', by(iso3)
		gen N = 1
		// Second, collapse to get the average # of actions per month, by country group
		collapse (sum) N (mean) `act_list', by(AE)
		foreach x of local act_list {
			// Average number of actions per year
			replace `x' = `x'*12
		}
			
		// Export to an Excel file
		export excel AE N `act_list' ///
			   using "../Data/iMaPP_Fig1234.xlsx", sheet("Act_`sheet'") firstrow(variables) missing("NA") sheetreplace
	restore	
*/		
}

*===============================================================================
* Step 3.B: LTV distribution (Figure 3)
*===============================================================================
if `flg_Fig_5' == 1 {

	use "../Data/iMaPP_M.dta", clear
	drop if ifscode == . 
	drop if AE      == .  // Curacao, whose AE/EMDE are not available
	sort  ifscode datem
	xtset ifscode datem

	// Keep only countires with LTV_average data
	egen sum_LTV_average = total(LTV_average), by(ifscode) missing
	gen flg_LTV_average  = (sum_LTV_average!=.)
	keep if flg_LTV_average == 1

    // Histogram of the average LTV limit, conditioning on the use of LTV limits (i.e., LTV_average < 100). 
	//   Please note that Netherelands is an exception, which has regulatory LTV limits 
	//   but at higher levels than 100 percent (its average LTV limit is at 102 percent as of Dec 2016).
	local End_date "2020M12" 
	label variable LTV_average "Average LTV limit (As of `End_date')"
	hist LTV_average if LTV_average<100 & datem == tm(`End_date'), frequency width(5) kdensity kdenopts(lwidth(1)) 
		graph export "./iMaPP_Fig3a.emf", as(emf) replace
	gen AE1 = (AE==0)
	graph box LTV_average if LTV_average<100 & datem == tm(`End_date'), over(AE1, relabel(1 "AEs" 2 "EMDEs"))
		graph export "./iMaPP_Fig3b.emf", as(emf) replace

	// Stats of the average LTV limit 
	xtsum LTV_average if datem == tm(`End_date')
	xtsum LTV_average if datem == tm(`End_date') & LTV_average < 100
	
	// Stats to report in Appendix I
	pwcorr LTV_average LTV_median
	gen dLTV_average = d.LTV_average
	gen dLTV_median  = d.LTV_median
	* br Country datem LTV_average dLTV_average dLTV_median
	
	// Number of changes: more changes in LTV_average than LTV_median 
	count if dLTV_median  != 0 & dLTV_median  != . & dLTV_average == 0
	count if dLTV_average != 0 & dLTV_average != . & dLTV_median == 0

	// LTV changes without moving its median: mostly tightening at risky segments
	count if dLTV_average != 0 & dLTV_average != . & dLTV_median == 0
	count if dLTV_average < 0  & dLTV_average != . & dLTV_median == 0
	count if dLTV_average > 0  & dLTV_average != . & dLTV_median == 0
	
	*----------------------------------------------------------
	* Summary stat in a Word document
	*----------------------------------------------------------
	// Install "xtsum2docx.ado", if it has not been installed yet 
	capture ssc install xtsum2docx

	// Name of the Word document
	local wordfile iMaPP_LTV_average_statistics.docx
	
	// Statistics at the end-2016, conditioning that it is lower than 100
	xtsum2docx LTV_average if LTV_average<100 & datem == tm(`End_date') using "./`wordfile'", replace ///
	 mean(%9.1f) min(%9.1f) max(%9.1f) p25(%9.1f) median(%9.1f) p75(%9.1f) obs(%9.0fc) xtn(%9.0fc) ///
	 order(mean min p25 median p75 max obs xtn) ///
	 title("Average LTV Limit (<100%) - All (as of `End_date')")
	xtsum2docx LTV_average if LTV_average<100 & datem == tm(`End_date') & AE == 1 using "./`wordfile'", append ///
	 mean(%9.1f) min(%9.1f) max(%9.1f) p25(%9.1f) median(%9.1f) p75(%9.1f) obs(%9.0fc) xtn(%9.0fc) ///
	 order(mean min p25 median p75 max obs xtn) ///
	 title("Average LTV Limit (<100%) - AEs (as of `End_date')")
	xtsum2docx LTV_average if LTV_average<100 & datem == tm(`End_date') & AE == 0 using "./`wordfile'", append ///
	 mean(%9.1f) min(%9.1f) max(%9.1f) p25(%9.1f) median(%9.1f) p75(%9.1f) obs(%9.0fc) xtn(%9.0fc) ///
	 order(mean min p25 median p75 max obs xtn) ///
	 title("Average LTV Limit (<100%) - EMDEs (as of `End_date')")
		
}


*===============================================================================
* Step 4: MaPP and Credit Growth(Figure 4)
*===============================================================================
if `flg_Fig_4' == 1 {
	// Load data
	/*
	use "C:\Users\MNarita\Box Sync\iMaPP WP\Data\quarterly_wp_macropru_final5.dta", clear
	xtset ifscode quarter
	sort ifscode quarter
	rename quarter dateq
	*/
	
	*use "../Data/quarterly_wp_macropru_final8_iMaPP2021_ex.dta", clear
	use "../Data/quarterly_wp_macropru_final8_iMaPP2021.dta", clear
 	xtset ifscode dateq
	sort ifscode dateq
	
	// Number of net tightening
	gen allsum_yoy_s = allsum_yoy*4
	label var allsum_yoy_s "# of net tightening, all 17 measures"
	
	/*
	***** Check 1 ***** 
		local instrument_list ccb conservation capital lvr llp lcg loanr lfc ///
							  ltv dsti tax lr ltd lfx rr sifi ot
		gen allsum_check = 0
		foreach x of local instrument_list {
			replace allsum_check = allsum_check + `x'
		}	
		gen allsum_diff = allsum_yoy_s - allsum_yoy_s
		su allsum_diff allsum_yoy_s allsum_yoy_s

	***** Check 2 ***** 
		bysort dateq: egen allsum_yoy_EM = sum(allsum_yoy_s) if EM==1
		bysort dateq: egen allsum_yoy_AE = sum(allsum_yoy_s) if AE==1	
		
		collapse (sum) allsum_yoy_s (firstnm) allsum_yoy_EM allsum_yoy_AE, by(dateq AE EMDE)
		gen diff_AE = allsum_yoy_s -  allsum_yoy_AE
		gen diff_EM = allsum_yoy_s -  allsum_yoy_EM
		su diff*
	*/
	*xtsum d4x_lnrhhdebtexbm_w1 if AE == 1
	*xtsum d4x_lnrhhdebtexbm_w1 if AE == 0
	xtsum d4lnrhhdebt_w1 if AE == 1
	xtsum d4lnrhhdebt_w1 if AE == 0
	
	collapse (sum) allsum_ = allsum_yoy_s (mean) d4lnrhhdebt_w1_ = d4lnrhhdebt_w1, by(dateq AE)
	*collapse (sum) allsum_ = allsum_yoy_s (mean) d4lnrhhdebt_w1_ = d4x_lnrhhdebtexbm_w1, by(dateq AE)
	sort AE dateq
	tsline allsum_ if AE == 1
	tsline allsum_ if AE == 0
	
	// Reshape the dataset to long
	gen 	C_group = "AE" if AE == 1
	replace C_group = "EM" if AE == 0
	drop AE
	reshape wide allsum_ d4lnrhhdebt_w1_, i(dateq) j(C_group) string
	label var allsum_AE "# of net tightening (AE, sum)"
	label var allsum_EM "# of net tightening (EMDE, sum)"	
	label var d4lnrhhdebt_w1_AE "Credit growth yoy (AE, mean)"
	label var d4lnrhhdebt_w1_EM "Credit growth yoy (EMDE, mean)"	
	
	// Export to an Excel file
	export excel dateq allsum_AE allsum_EM d4lnrhhdebt_w1_AE d4lnrhhdebt_w1_EM ///
			   using "../Data/Figure_Discriptive_2022-05-18.xlsx", sheet("Fig 4") firstrow(variables) missing("NA") sheetreplace
}


*===============================================================================
* Step 6: Summary Stats (Appendix II Table 3)
*===============================================================================
if `flg_Fig_Stats' == 1 {
	
	*----------------------------------------------------------
	* Load the estimation sample
	*----------------------------------------------------------
	// LTV variable to be used
	local LTV_Qvar LTV_Qend // LTV_Qmax, LTV_Qmean, or LTV_Qend

	// Sample restrictions
	local Country     ALL         // ALL EM AE
	local StartYear   1990        // 1990 2000
	local SmplPeriod  "year >= `StartYear' & year <= 2016"
	local runWithDataAfter = 1  // Data to use (0: Before or 1: after revision)
	local str_data_version 8_iMaPP2021	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"

	// Load the estimation data
	use "../Data/quarterly_wp_macropru_final`str_data_version'.dta", clear

	// Time variables
	*rename  quarter dateq
	capture gen qr = quarter(dofq(dateq))
	label var dateq "Time variable in Stata format (quarterly:%tq)"
	label var qr    "Quarter"
	xtset isocode dateq

	// Memo item
	gen dLTV_med = d.LTV_Qend_med
	label var dLTV_med "Change in the median LTV (supplemental info)"
	count if dltv_level != 0 & dltv_level < .
	count if dLTV_med != 0   & dLTV_med < .

	*----------------------------------------------------------
	* Summary statistics of the estimation sample
	*----------------------------------------------------------
	// Whole sample
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate allsum_yoy allloansum_yoy demand_side_yoy supply_side_all_yoy supply_loans_yoy supply_general_yoy supply_capital_yoy ccb_yoy conservation_yoy capital_yoy lvr_yoy llp_yoy lcg_yoy loanr_yoy lfc_yoy ltv_yoy dsti_yoy tax_yoy lr_yoy ltd_yoy lfx_yoy rr_yoy sifi_yoy ot_yoy dltv_level if SAMPLE_ALL_HHD ==1, stat(mean sd) column(stats)
	// AE sample					
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate allsum_yoy allloansum_yoy demand_side_yoy supply_side_all_yoy supply_loans_yoy supply_general_yoy supply_capital_yoy ccb_yoy conservation_yoy capital_yoy lvr_yoy llp_yoy lcg_yoy loanr_yoy lfc_yoy ltv_yoy dsti_yoy tax_yoy lr_yoy ltd_yoy lfx_yoy rr_yoy sifi_yoy ot_yoy dltv_level if SAMPLE_AE_HHD ==1, stat(mean sd) column(stats)
	// EMDE sample	
	tabstat d4lnrhhdebt_w1 d4lnrpvcon_w1 d4lnrp_house_w1 d4lnrgdp_w1 strate allsum_yoy allloansum_yoy demand_side_yoy supply_side_all_yoy supply_loans_yoy supply_general_yoy supply_capital_yoy ccb_yoy conservation_yoy capital_yoy lvr_yoy llp_yoy lcg_yoy loanr_yoy lfc_yoy ltv_yoy dsti_yoy tax_yoy lr_yoy ltd_yoy lfx_yoy rr_yoy sifi_yoy ot_yoy dltv_level if SAMPLE_EM_HHD ==1, stat(mean sd) column(stats)			

}
/*
*===============================================================================
* Step 7: LTV introduction cases in the estimation sample (Quarterly)
*===============================================================================
if `flg_LTV_intro' == 1 {
	
	*----------------------------------------------------------
	* Load the estimation sample
	*----------------------------------------------------------
	// LTV variable to be used
	local LTV_Qvar LTV_Qend // LTV_Qmax, LTV_Qmean, or LTV_Qend

	// Sample restrictions
	local Country     ALL         // ALL EM AE
	local StartYear   1990        // 1990 2000
	local SmplPeriod  "year >= `StartYear' & year <= 2016"
	local runWithDataAfter = 1  // Data to use (0: Before or 1: after revision)
	// local str_data_version 8_iMaPP2019	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"
	local str_data_version 8_iMaPP2021	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"
	
	// Load the estimation data
	local estimation_data = 0
	if `estimation_data' == 1 {
		use "../Data/quarterly_wp_macropru_final`str_data_version'.dta", clear
	}
	else {
		use "../Data/iMaPP_Q.dta", clear
		rename Year year
	}
	
	// Time variables
	*rename  quarter dateq
	capture gen qr = quarter(dofq(dateq))
	label var dateq "Time variable in Stata format (quarterly:%tq)"
	label var qr    "Quarter"
	xtset ifscode dateq

	// dltv_level variable to use
	if `runWithDataAfter' == 1 {
		capture drop ltv_level 
		capture drop dltv_level
		gen 	  ltv_level  = `LTV_Qvar'
		gen       dltv_level = d.ltv_level
		label var ltv_level  "`LTV_Qvar': Quarterly aggregation of LTV_average"
		label var dltv_level "d.`LTV_Qvar': Quarterly aggregation of LTV_average"
	}
	
	/*
	// Natural log of LTV level
	gen lnltv_level = ln(ltv_level)
	label var lnltv_level  "log of `LTV_Qvar': Quarterly aggregation of LTV_average"
	gen dlnltv_level = 100*(lnltv_level - l.lnltv_level)
	label var dlnltv_level  "d. log of `LTV_Qvar': Quarterly aggregation of LTV_average"
	*/
	
	// LTV level in the previous quarter
	capture drop L_LTV
	gen L_LTV = l.ltv_level
	label var L_LTV "Lag of ltv_level"	
	
	// Dummy variable to indicate an introduction of LTV limits (used to exlcude these cases)
	// For Malaysia and Romania, there are two incidences of LTV introductions 
	// because the limits were once abolished (intead of being set at 100) but re-introduced later.
	egen ltv_max_by_country = max(ltv_level), by(ifscode)
	gen byte dltv_intro = (L_LTV == ltv_max_by_country & 100 <= ltv_max_by_country & dltv_level < 0)
	label var dltv_intro "Dummy variable: 1 indicates an introduction of LTV limits"
	egen N_dltv_intro_by_country = sum(dltv_intro), by(ifscode)
	
	*----------------------------------------------------------
	* Summary statistics
	*----------------------------------------------------------
	// Summary stats
	su ltv_level dltv_intro if !missing(ltv_level) & `SmplPeriod'
	su ltv_level dltv_intro if !missing(ltv_level) & `SmplPeriod' & dltv_intro == 1
	su ltv_level if !missing(ltv_level) & dateq == tq(2016q4) & ltv_level < 100, detail
	su ltv_level if !missing(ltv_level) & dateq == tq(2020q4) & ltv_level < 100, detail
	
	count if !missing(ltv_level) & dateq == tq(2016q4) & ltv_level < 100
	count if !missing(ltv_level) & dltv_intro == 1
	
	br Country dateq L_LTV ltv_level dltv_level dltv_intro if dltv_intro == 1
	tsline ltv_level if Country == "Serbia"
	br Country iso2 dateq ltv_level N_dltv_intro_by_country if  dateq == tq(2016q4)  &  !missing(ltv_level)
	br Country iso2 dateq ltv_level N_dltv_intro_by_country if  dateq == tq(2020q4)  &  !missing(ltv_level)
}

***********  END *************
log close
exit
