/*

Estimation_initial_LTV.do

Notes: This is the codes for the AIPW estimation for Section IV of the following paper.

	   Zohair Alam, Adrian Alter, Jesse Eiseman, Gaston Gelos, Heedon Kang, Machiko Narita, Erlend Nier, and Naixi Wang
	   "Digging Deeper – Evidence on the Effects of Macroprudential Policies from a New Database"   
	   
Last Updated: April 24, 2022 (Reflecting JMBC referee's comments)
	- Add house price and other MaPP as controls)
	- Streamlined the name of the output files, by dropping the elements that have been settled (e.g., Lag0, ALL, LTV_Qend). 
	  See the notes on Apr 23, 2022 of Notes.docx.
	- Datasets:
		- quarterly_wp_macropru_final5.dta: Data with the original iMaPP (WP version)
		- quarterly_wp_macropru_final6.dta: Data with the 2019 iMaPP		
		- quarterly_wp_macropru_final61.dta: Data with the 2019 iMaPP with Adrian's mortgage data
		- quarterly_wp_macropru_final7.dta: Data with the 2020 iMaPP with Adrian's mortgage data (used in Sep 5, 2021)
		- quarterly_wp_macropru_final8_iMaPP2019.dta: Data with the 2019 iMaPP with Adrian's mortgage data & comprehensive mortgage data

Last Updated: August 25, 2022  (fix the way to handle nonintro cases)
Previous version: May 27, 2022

*/

*===============================================================================
* Step 1: Set up
*===============================================================================
//cd "C:\Users\mnarita\Box Sync\iMaPP WP\Codes"	
clear all 
set maxvar 10000
set more off
set excelxlsxlargefile on
pause off
capture log close

// LTV variable to be used
local LTV_Qvar LTV_Qend // LTV_Qmax, LTV_Qmean, or LTV_Qend

// Set threshold for high level of LTV by country group
local high_AE    75
local high_EMDE  75

// Set threshold to exclude extreme LTV changes
local extreme_T 25
local extreme_L 25

// Select whether to include HP and OMaPP as controls
local flg_ctrl_lag_house_prices = 1 // Set 1 if the lag of house prices is included.
local flg_ctrl_other_policies   = 1 // Set 1 if other MaPP policies (excl. LTV) is included.

local flg_lnLTV    = 1 // Set 1 to take the naturala logarithm of LTV to examine the effect of 1 percent change, instead of 1 ppt.
local flg_nonIntro = 1 // Set 1 to exclude obs with L_LTV >= 100, i.e., introduction of LTV

// For T20 group, introduction cases account for 9 out of the 13 cases where bank loans etc. available.
// Among the remaining 4, two of them (Malaysia 2011Q4, Poland 2014Q2) are found to be LTV limits on loans for "commercial residential estates".
// To drop these cases for all estimations, use the following flag variable:
local flg_nonCRE = 1 // Set 1 to exclude CRE LTV tightening (Hong Kong 1994Q1 and 2013Q1, Lebanon 2008Q3, Malaysia 2011Q4, Poland 2014Q2).

// Sample restrictions
local Country     ALL         // ALL EM AE
local StartYear   1990        // 1990 2000
local SmplPeriod  "year >= `StartYear' & year <= 2016"
local runWithDataAfter = 1  // Data to use (0: Before or 1: after revision)
// local str_data_version 8_iMaPP2019	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"
local str_data_version 8_iMaPP2021	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"
if `runWithDataAfter'==0 local str_data_version 5	// The "before" data file is "quarterly_wp_macropru_final5.dta"

// Spec name: 
// Example: T1025_L10_in0_2000
// local group_T 10
// local group_L 10
// local extreme_T 25
// local extreme_L 
// local StartYear 2000 
local Spec "`high_AE'_`high_EMDE'_T`extreme_T'L`extreme_L'_`StartYear'"
di "Spec = `Spec'"

// Other controls
local controls_FE
local Controls ""
if `flg_ctrl_lag_house_prices'==1 {	
	// Includ house prices as a control
	local controls_FE `controls_FE' l.d4lnrp_house_w1
	local Controls `Controls'_HP
}
if `flg_ctrl_other_policies'==1 {	
	// Includ other MaPP policies (excl. LTV) as a control: Average of the dummies from t to t-3.
// 	local controls_FE `controls_FE' l.allsum_wo_ltv  l2.allsum_wo_ltv  l3.allsum_wo_ltv  l4.allsum_wo_ltv
// 	local Controls `Controls'_OMaPP4
	local controls_FE `controls_FE' l.allsum_wo_ltv_yoy 
	local Controls `Controls'_OMaPP
}

if `flg_lnLTV'==1 local LnLTV "_lnLTV"

local NonIntro ""
if `flg_nonIntro'==1 {
	local NonIntro "_NonIntro"
}

local NonCRE ""
if `flg_nonCRE'==1 {
	local NonCRE "_NonCRE"
}


// Local variable that summarizes sample to be used:
local touse `SmplPeriod' 
di "touse = `touse'"

// Set the log file
local today : display %tdCYND date(c(current_date), "DMY")  // Date (YYYYMMDD)
log using "../Results/Macropru/LTV/Log_Initial_LTV_`Spec'_`today'`Controls'`LnLTV'`NonIntro'`NonCRE'_v`str_data_version'.txt", text replace

di "Spec = `Spec'"
di "touse = `touse'"

*-------------------------------------------------------------------------------
* Load data
*-------------------------------------------------------------------------------
local ltv_related_var_list `dltv_var' 
if `runWithDataAfter' == 1 {
	*use "../Data/quarterly_wp_macropru_final6.dta", clear // iMaPP-2019
// 	use "../Data/quarterly_wp_macropru_final61.dta", clear
	use "../Data/quarterly_wp_macropru_final`str_data_version'.dta", clear
}
else {
	//  use "\\data1\MCM\DRAFTS\MP\iMaPP Database\2 iMaPP - Master\2019 update\2 Update\3 Checks\3 Analysis\Data\quarterly_wp_macropru_final6.dta", clear
	use "C:\Users\MNarita\Box Sync\iMaPP WP\Data\quarterly_wp_macropru_final5.dta", clear
}

*-------------------------------------------------------------------------------
* Time variables
*-------------------------------------------------------------------------------
*rename  quarter dateq
capture gen qr = quarter(dofq(dateq))
label var dateq "Time variable in Stata format (quarterly:%tq)"
label var qr    "Quarter"
xtset isocode dateq

// For comparison, drop observations in the first quarter
if `runWithDataAfter' {
	foreach v of local ltv_related_var_list {
		replace `v' = . if dateq <= yq(`StartYear',1)
	}
}
capture drop _Iquarter_*

// Keep the selected sample only
di as err "Sample: `Country' +++++++++++ (cmd:keep if `Country' == 1) +++++++++++" 
su ALL AE EM
keep if `Country'   == 1
su ALL AE EM if ALL == 1 
su ALL AE EM if AE  == 1 
su ALL AE EM if EM  == 1 

*-------------------------------------------------------------------------------
* dltv_level variables
*-------------------------------------------------------------------------------
// dltv_level variable to use
capture drop ltv_level 
capture drop dltv_level
gen 	  ltv_level  = `LTV_Qvar'
gen       dltv_level = d.ltv_level
label var ltv_level  "`LTV_Qvar': Quarterly aggregation of LTV_average"
label var dltv_level "d.`LTV_Qvar': Quarterly aggregation of LTV_average"

// LTV level in the previous quarter
capture drop L_LTV
gen L_LTV = l.ltv_level
label var L_LTV "Lag of ltv_level"

// Dummy variable to indicate an introduction of LTV limits (used to exlcude these cases)
// For Malaysia and Romania, there are two incidences of LTV introductions 
// because the limits were once abolished (intead of being set at 100) but re-introduced later.
egen ltv_max_by_country = max(ltv_level), by(ifscode)
gen byte dltv_intro = (L_LTV == ltv_max_by_country & 100 <= ltv_max_by_country & dltv_level < 0)
label var dltv_intro "Dummy variable: 1 indicates an introduction of LTV limits"

if `flg_nonIntro'==1 {
	replace ltv_level  = . if dltv_intro == 1
	replace dltv_level = . if dltv_intro == 1
}
if `flg_nonCRE'==1 {
	// Exclude CRE LTV tightening (Hong Kong 1994Q1 and 2013Q1, Lebanon 2008Q3, Malaysia 2011Q4, Poland 2014Q2).
	replace ltv_level  = . if iso3 == "HKG" & dateq == yq(1994,1)
	replace ltv_level  = . if iso3 == "HKG" & dateq == yq(2013,1)
 	*replace ltv_level  = . if iso3 == "LBN" & dateq == yq(2008,3) // Not in the estimation sample
	replace ltv_level  = . if iso3 == "MYS" & dateq == yq(2011,4)
	replace ltv_level  = . if iso3 == "POL" & dateq == yq(2014,2)
	
	replace dltv_level = . if iso3 == "HKG" & dateq == yq(1994,1)
	replace dltv_level = . if iso3 == "HKG" & dateq == yq(2013,1)
 	*replace dltv_level = . if iso3 == "LBN" & dateq == yq(2008,3) // Not in the estimation sample
	replace dltv_level = . if iso3 == "MYS" & dateq == yq(2011,4)
	replace dltv_level = . if iso3 == "POL" & dateq == yq(2014,2)
}

// Trim outliers if specified
if "`extreme_T'" != "" {
	replace ltv_level  = . if -`extreme_T' > dltv_level // "-25 > dltv_level" to exclude very large tightening
	replace dltv_level = . if -`extreme_T' > dltv_level // "-25 > dltv_level" to exclude very large tightening	
}
if "`extreme_L'" != "" {
	replace ltv_level  = . if dltv_level > `extreme_L' // "dltv_level > 25" to exclude very large loosening
	replace dltv_level = . if dltv_level > `extreme_L' // "dltv_level > 25" to exclude very large loosening	
}

// Natural log of LTV level
gen lnltv_level = ln(ltv_level)
label var lnltv_level  "log of `LTV_Qvar': Quarterly aggregation of LTV_average"
gen dlnltv_level = 100*(lnltv_level - l.lnltv_level)
label var dlnltv_level  "d. log of `LTV_Qvar': Quarterly aggregation of LTV_average"

****Interaction with the level******
**95 and 90 is the top25 form ltv_level<100 across EMs and AEs, respectively
capture drop ltv_level_high

gen     ltv_level_high=1 if AE==1 & L_LTV>=`high_AE'   & !missing(L_LTV)
replace ltv_level_high=1 if EM==1 & L_LTV>=`high_EMDE' & !missing(L_LTV)
replace ltv_level_high=0 if !missing(L_LTV) & missing(ltv_level_high)

// Winsorize dltv_level
capture drop   dltv_level_w1*
winsor    dltv_level if dltv_level ~=0 , gen(dltv_level_w1) p(0.1)
replace   dltv_level_w1 = 0 if dltv_level ==0
label var dltv_level_w1 "dltv_level_w, Winsorized fraction .1"

/*
// Summary table to a Word file
xtsum2docx ltv_level if ltv_level<0 using test.docx, ///
 replace mean(%9.1f) p25(%9.1f) median(%9.1f) p75(%9.1f) obs(%9.0fc) xtn(%9.0fc) order(mean p25 median p75 obs xtn)
*/

*-------------------------------------------------------------------------------
* Other variables
*-------------------------------------------------------------------------------
// Rename the mortgage variables from Adrian to keep the order in the result file
rename lnrmortdebt         w_lnrmortdebt 
rename d4lnrmortdebt       d4w_lnrmortdebt
rename d4lnrmortdebt_w1    d4w_lnrmortdebt_w1 
replace d4w_lnrmortdebt    = 100*d4w_lnrmortdebt
replace d4w_lnrmortdebt_w1 = 100*d4w_lnrmortdebt_w1
label var w_lnrmortdebt   "log of mortgage debt by Adrian"
label var d4w_lnrmortdebt "Y-on-Y change (x-l4.x): x is w_lnrmortdebt by Adrian"	
label var d4w_lnrmortdebt_w1 "d4w_lnrmortdebt: winsorized at 0.01"	

rename lnrnonmortdebt         w_lnrnonmortdebt
rename d4lnrnonmortdebt       d4w_lnrnonmortdebt
rename d4lnrnonmortdebt_w1    d4w_lnrnonmortdebt_w1
replace d4w_lnrnonmortdebt    = 100*d4w_lnrnonmortdebt
replace d4w_lnrnonmortdebt_w1 = 100*d4w_lnrnonmortdebt_w1
label var w_lnrnonmortdebt   "log of non-mortgage debt by Adrian"
label var d4w_lnrnonmortdebt "Y-on-Y change (x-l4.x): x is w_lnrnonmortdebt by Adrian"	
label var d4w_lnrnonmortdebt_w1 "d4w_lnrnonmortdebt: winsorized at 0.01"	


// Balanced bank data series
gen x_bank_balanced = 1 if !missing(x_lnrbankloan) & !missing(x_lnrbankmort) & !missing(x_lnrbanknonmort)
gen d4x_bank_w1_balanced = 1 if !missing(d4x_lnrbankloan_w1) & !missing(d4x_lnrbankmort_w1) & !missing(d4x_lnrbanknonmort_w1)
local bankloan_list x_lnrbankloan x_lnrbankmort x_lnrbanknonmort
foreach v of local bankloan_list {
	gen `v'_b = `v' if x_bank_balanced == 1
	gen d4`v'_b_w1 = d4`v'_w1 if d4x_bank_w1_balanced == 1	
}

// Share data
gen x_share_m_hhdebt = 100*(exp(w_lnrmortdebt)*cpi)/hhdebt
gen x_share_b_hhdebt  = 100*x_bankloan/hhdebt
gen x_share_bm_bloan  = 100*x_bankmort/x_bankloan
label var x_share_m_hhdebt "Mortgage loans in percent of household debt"
label var x_share_b_hhdebt "Bank loans in percent of household debt"	
label var x_share_bm_bloan "Bank mortgage loans in percent of household loans by banks"

local share_varlist x_share_m_hhdebt x_share_b_hhdebt x_share_bm_bloan
foreach v of local share_varlist {
	replace `v' = . if `v' < 0 | `v' > 100
	local lab_tmp : variable label `v'
    qui gen d4`v' = `v' - L4.`v'
	label variable d4`v' "d4 `lab_tmp'"
	
	// Winsorize
	winsor d4`v', gen(d4`v'_w1) p(0.01)	
}


// Inflation
capture drop d1lncpi
capture drop d4lncpi
gen d1lncpi = 100*(lncpi-l1.lncpi)
gen d4lncpi = 100*(lncpi-l4.lncpi)
di d1lncpi "Inflation (Q-on-Q % change: 100*(lncpi-l1.lncpi)"
di d4lncpi "Inflation (Y-on-Y % change: 100*(lncpi-l4.lncpi)"

// Winsorize
capture drop d4lncpi_w1
winsor d4lncpi, gen(d4lncpi_w1) p(0.01)
label var d4lncpi_w1 "d4lncpi: winsorized at 0.01"	

*===============================================================================
* Estimation
*===============================================================================
// Output excel file name
local myfile="../Results/Macropru/LTV/Initial_LTV_`Spec'_`today'`Controls'`LnLTV'`NonIntro'`NonCRE'_v`str_data_version'.xls"

//[Outcome variable] Note: d4 and _w1 will be automatically added below, e.g., d4lnrhhdebt_w1 d4lnrpvcon_w1
local y_list lnrhhdebt lnrpvcon lnrgdp lnrp_house lncpi ///
	x_lnrbankmort x_lnrhhdebtexbm ///
	x_lnrbankloan x_lnrnonbankloan x_lnrbanknonmort ///
	w_lnrmortdebt w_lnrnonmortdebt ///
	x_lnrbankloan_b x_lnrbankmort_b x_lnrbanknonmort_b 
    *x_share_b_hhdebt x_share_bm_bloan x_share_m_hhdebt 

// Specify the list of country groups
local condlist ALL AE EM 

local app_or_rep replace

local dltv dltv_level
if `flg_lnLTV' local dltv dlnltv_level

foreach y of local y_list {
	foreach grp of local condlist {	
		
		// FE with interaction terms between initial high LTV level dummy and `dltv'	
		xi: areg d4`y'_w1 l.d4`y'_w1 l.strate l.d4lnrgdp_w1 `controls_FE' c.l.`dltv'##i.ltv_level_high c.l2.`dltv'##i.ltv_level_high c.l3.`dltv'##i.ltv_level_high c.l4.`dltv'##i.ltv_level_high i.dateq l.ltv_level if `touse' & `grp'==1 , absorb(ifscode) clu(ifscode)
		
		// Cumulative effect for the case of low initial LTV levels (lag)		
		lincom l.`dltv' + l2.`dltv' + l3.`dltv' +l4.`dltv' 
		local lincom_p_low   = r(p)
		local lincom_est_low = r(estimate)
				
		// Cumulative effect for the case of high initial LTV levels (lag)
		lincom l.`dltv' + l2.`dltv' + l3.`dltv' +l4.`dltv' ///
			+ 1.ltv_level_high#cL1.`dltv' + 1.ltv_level_high#cL2.`dltv' ///
			+ 1.ltv_level_high#cL3.`dltv' + 1.ltv_level_high#cL4.`dltv'	
		local lincom_p_high   = r(p)
		local lincom_est_high = r(estimate)
		
		// Difference between high and low initial LTV levels (lag)
		lincom 1.ltv_level_high#cL1.`dltv' + 1.ltv_level_high#cL2.`dltv' ///
			+ 1.ltv_level_high#cL3.`dltv' + 1.ltv_level_high#cL4.`dltv'	
		local lincom_p_diff   = r(p)
		local lincom_est_diff = r(estimate)
						
		// Export results to the excel file
		outreg2  using `myfile', excel `app_or_rep' keep(l.d4`y'_w1 l.strate l.d4lnrgdp_w1 `controls_FE' c.l.`dltv'##i.ltv_level_high c.l2.`dltv'##i.ltv_level_high c.l3.`dltv'##i.ltv_level_high c.l4.`dltv'##i.ltv_level_high) addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, Xtreg, Period, >=`StartYear', Group, `grp', Cum eff low ini LTV, `lincom_est_low', P-value low ini LTV, `lincom_p_low', Cum eff high ini LTV, `lincom_est_high', P-value high ini LTV, `lincom_p_high', Diff high vs low, `lincom_est_diff', P-value Diff high vs low, `lincom_p_diff') label		
		if "`app_or_rep'" == "replace" local app_or_rep append
		
	}
}

// Simple correlation between initial levels vs. changes
local ltv ltv_level
if `flg_lnLTV' local ltv lnltv_level
// Overall
pwcorr l.`ltv' `dltv', star(0.05)
// When tightening
pwcorr l.`ltv' `dltv' if `dltv' < 0, star(0.05)
// When loosening
pwcorr l.`ltv' `dltv' if `dltv' > 0, star(0.05)

**********************************
log close
exit
