/*

Estimation_subsample.do

Notes: This is the codes for the subsample estimations for Section IV.C of the following paper.

	   Zohair Alam, Adrian Alter, Jesse Eiseman, Gaston Gelos, Heedon Kang, Machiko Narita, Erlend Nier, and Naixi Wang
	   "Digging Deeper – Evidence on the Effects of Macroprudential Policies from a New Database"   
	   
Last Updated: April 24, 2023 (Reflecting JMBC referee's comments)

		- quarterly_wp_macropru_final8_iMaPP2019.dta: Data with the 2019 iMaPP with Adrian's mortgage data & comprehensive mortgage data
	
Last Updated: April 25, 2023
Previous version: April 26, 2022
Original version: Excerpt from macropru_server_wp4_aa_rob-v2.do

*/

*===============================================================================
* Step 1: Set up
*===============================================================================
//cd "C:\Users\mnarita\Box Sync\iMaPP WP\Codes"	
clear all 
set maxvar 10000
set more off
set excelxlsxlargefile on
pause off
capture log close

// Which part to run
local run_subsample_OLS = 0
local run_subsample_GMM = 0
local run_slope_dummies = 1

// LTV variable to be used
local LTV_Qvar LTV_Qend // LTV_Qmax, LTV_Qmean, or LTV_Qend

// Set threshold for high level of LTV by country group
local high_AE    75
local high_EMDE  75

// Set threshold to exclude extreme LTV changes
local extreme_T 25
local extreme_L 25

// Select whether to include HP and OMaPP as controls
local flg_ctrl_lag_house_prices = 1 // Set 1 if the lag of house prices is included.
local flg_ctrl_other_policies   = 1 // Set 1 if other MaPP policies (excl. LTV) is included.

local flg_lnLTV = 0 // Set 1 to take the naturala logarithm of LTV to examine the effect of 1 percent change, instead of 1 ppt.
local flg_nonIntro = 0 // Set 1 to exclude obs with L_LTV >= 100, i.e., introduction of LTV

// For T20 group, introduction cases account for 9 out of the 13 cases where bank loans etc. available.
// Among the remaining 4, two of them (Malaysia 2011Q4, Poland 2014Q2) are found to be LTV limits on loans for "commercial residential estates".
// To drop these cases for all estimations, use the following flag variable:
local flg_nonCRE = 1 // Set 1 to exclude CRE LTV tightening (Hong Kong 1994Q1 and 2013Q1, Lebanon 2008Q3, Malaysia 2011Q4, Poland 2014Q2).

// Sample restrictions
local Country     ALL         // ALL EM AE
local StartYear   1990        // 1990 2000
local SmplPeriod  "year >= `StartYear' & year <= 2016"

local runWithDataAfter = 1  // Data to use (0: Before or 1: after revision)
// local str_data_version 8_iMaPP2019	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"
local str_data_version 8_iMaPP2021	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"

if `runWithDataAfter'==0 local str_data_version 5	// The "before" data file is "quarterly_wp_macropru_final5.dta"
if `runWithDataAfter'==0 local StartYear 2000		// The "before" data starts from the year 2000

// Spec name: 
// Example: T25L25_2000
// local extreme_T 25
// local extreme_L 25
// local StartYear 2000 
local Spec "T`extreme_T'L`extreme_L'_`StartYear'"
di "Spec = `Spec'"

// Other controls
local controls_FE
local Controls ""
if `flg_ctrl_lag_house_prices'==1 {	
	// Includ house prices as a control
	local controls_FE `controls_FE' l.d4lnrp_house_w1
	local Controls `Controls'_HP
}
if `flg_ctrl_other_policies'==1 {	
	// Includ other MaPP policies (excl. LTV) as a control: Average of the dummies from t to t-3.
// 	local controls_FE `controls_FE' l.allsum_wo_ltv  l2.allsum_wo_ltv  l3.allsum_wo_ltv  l4.allsum_wo_ltv
// 	local Controls `Controls'_OMaPP4
	local controls_FE `controls_FE' l.allsum_wo_ltv_yoy 
	local Controls `Controls'_OMaPP
}

if `flg_lnLTV'==1 local LnLTV "_lnLTV"

local NonIntro ""
if `flg_nonIntro'==1 {
	local NonIntro "_NonIntro"
}

local NonCRE ""
if `flg_nonCRE'==1 {
	local NonCRE "_NonCRE"
}


// Local variable that summarizes sample to be used:
local touse `SmplPeriod' 
di "touse = `touse'"

// Set the log file
local today : display %tdCYND date(c(current_date), "DMY")  // Date (YYYYMMDD)
log using "../Results/Macropru/LTV/Log_Subsample_`Spec'_`today'`Controls'`LnLTV'`NonIntro'`NonCRE'_v`str_data_version'.txt", text replace

di "Spec = `Spec'"
di "touse = `touse'"

*-------------------------------------------------------------------------------
* Load data
*-------------------------------------------------------------------------------
local ltv_related_var_list `dltv_var' 
use "../Data/quarterly_wp_macropru_final`str_data_version'.dta", clear

*-------------------------------------------------------------------------------
* Time variables
*-------------------------------------------------------------------------------
if `runWithDataAfter'==0 rename quarter dateq
capture gen qr = quarter(dofq(dateq))
label var dateq "Time variable in Stata format (quarterly:%tq)"
label var qr    "Quarter"
xtset isocode dateq

// For comparison, drop observations in the first quarter
if `runWithDataAfter' {
	foreach v of local ltv_related_var_list {
		replace `v' = . if dateq <= yq(`StartYear',1)
	}
}
capture drop _Iquarter_*

// Keep the selected sample only
di as err "Sample: `Country' +++++++++++ (cmd:keep if `Country' == 1) +++++++++++" 
su ALL AE EM
keep if `Country'   == 1
su ALL AE EM if ALL == 1 
su ALL AE EM if AE  == 1 
su ALL AE EM if EM  == 1 

*-------------------------------------------------------------------------------
* dltv_level variables
*-------------------------------------------------------------------------------
// dltv_level variable to use
capture drop ltv_level 
capture drop dltv_level
gen 	  ltv_level  = `LTV_Qvar'
gen       dltv_level = d.ltv_level
label var ltv_level  "`LTV_Qvar': Quarterly aggregation of LTV_average"
label var dltv_level "d.`LTV_Qvar': Quarterly aggregation of LTV_average"

// LTV level in the previous quarter
capture drop L_LTV
gen L_LTV = l.ltv_level
label var L_LTV "Lag of ltv_level"

// Dummy variable to indicate an introduction of LTV limits (used to exlcude these cases)
// For Malaysia and Romania, there are two incidences of LTV introductions 
// because the limits were once abolished (intead of being set at 100) but re-introduced later.
egen ltv_max_by_country = max(ltv_level), by(ifscode)
gen byte dltv_intro = (L_LTV == ltv_max_by_country & 100 <= ltv_max_by_country & dltv_level < 0)
label var dltv_intro "Dummy variable: 1 indicates an introduction of LTV limits"

if `flg_nonIntro'==1 {
	replace ltv_level  = . if dltv_intro == 1
	replace dltv_level = . if dltv_intro == 1
}
if `flg_nonCRE'==1 {
	// Exclude CRE LTV tightening (Hong Kong 1994Q1 and 2013Q1, Lebanon 2008Q3, Malaysia 2011Q4, Poland 2014Q2).
	replace ltv_level  = . if iso3 == "HKG" & dateq == yq(1994,1)
	replace ltv_level  = . if iso3 == "HKG" & dateq == yq(2013,1)
 	*replace ltv_level  = . if iso3 == "LBN" & dateq == yq(2008,3) // Not in the estimation sample
	replace ltv_level  = . if iso3 == "MYS" & dateq == yq(2011,4)
	replace ltv_level  = . if iso3 == "POL" & dateq == yq(2014,2)
	
	replace dltv_level = . if iso3 == "HKG" & dateq == yq(1994,1)
	replace dltv_level = . if iso3 == "HKG" & dateq == yq(2013,1)
 	*replace dltv_level = . if iso3 == "LBN" & dateq == yq(2008,3) // Not in the estimation sample
	replace dltv_level = . if iso3 == "MYS" & dateq == yq(2011,4)
	replace dltv_level = . if iso3 == "POL" & dateq == yq(2014,2)
}

// Trim outliers if specified
if "`extreme_T'" != "" {
	replace ltv_level  = . if -`extreme_T' > dltv_level // "-25 > dltv_level" to exclude very large tightening
	replace dltv_level = . if -`extreme_T' > dltv_level // "-25 > dltv_level" to exclude very large tightening	
}
if "`extreme_L'" != "" {
	replace ltv_level  = . if dltv_level > `extreme_L' // "dltv_level > 25" to exclude very large loosening
	replace dltv_level = . if dltv_level > `extreme_L' // "dltv_level > 25" to exclude very large loosening	
}

// Natural log of LTV level
gen lnltv_level = ln(ltv_level)
label var lnltv_level  "log of `LTV_Qvar': Quarterly aggregation of LTV_average"
gen dlnltv_level = 100*(lnltv_level - l.lnltv_level)
label var dlnltv_level  "d. log of `LTV_Qvar': Quarterly aggregation of LTV_average"

****Interaction with the level******
**95 and 90 is the top25 form ltv_level<100 across EMs and AEs, respectively
capture drop ltv_level_high

gen     ltv_level_high=1 if AE==1 & L_LTV>=`high_AE'   & !missing(L_LTV)
replace ltv_level_high=1 if EM==1 & L_LTV>=`high_EMDE' & !missing(L_LTV)
replace ltv_level_high=0 if !missing(L_LTV) & missing(ltv_level_high)

// Winsorize dltv_level
capture drop   dltv_level_w1*
winsor    dltv_level if dltv_level ~=0 , gen(dltv_level_w1) p(0.1)
replace   dltv_level_w1 = 0 if dltv_level ==0
label var dltv_level_w1 "dltv_level_w, Winsorized fraction .1"

/*
// Summary table to a Word file
xtsum2docx ltv_level if ltv_level<0 using test.docx, ///
 replace mean(%9.1f) p25(%9.1f) median(%9.1f) p75(%9.1f) obs(%9.0fc) xtn(%9.0fc) order(mean p25 median p75 obs xtn)
*/

*-------------------------------------------------------------------------------
* Other variables
*-------------------------------------------------------------------------------
// Rename the mortgage variables from Adrian to keep the order in the result file
rename lnrmortdebt         w_lnrmortdebt 
rename d4lnrmortdebt       d4w_lnrmortdebt
rename d4lnrmortdebt_w1    d4w_lnrmortdebt_w1 
replace d4w_lnrmortdebt    = 100*d4w_lnrmortdebt
replace d4w_lnrmortdebt_w1 = 100*d4w_lnrmortdebt_w1
label var w_lnrmortdebt   "log of mortgage debt by Adrian"
label var d4w_lnrmortdebt "Y-on-Y change (x-l4.x): x is w_lnrmortdebt by Adrian"	
label var d4w_lnrmortdebt_w1 "d4w_lnrmortdebt: winsorized at 0.01"	

rename lnrnonmortdebt         w_lnrnonmortdebt
rename d4lnrnonmortdebt       d4w_lnrnonmortdebt
rename d4lnrnonmortdebt_w1    d4w_lnrnonmortdebt_w1
replace d4w_lnrnonmortdebt    = 100*d4w_lnrnonmortdebt
replace d4w_lnrnonmortdebt_w1 = 100*d4w_lnrnonmortdebt_w1
label var w_lnrnonmortdebt   "log of non-mortgage debt by Adrian"
label var d4w_lnrnonmortdebt "Y-on-Y change (x-l4.x): x is w_lnrnonmortdebt by Adrian"	
label var d4w_lnrnonmortdebt_w1 "d4w_lnrnonmortdebt: winsorized at 0.01"	


// Balanced bank data series
gen x_bank_balanced = 1 if !missing(x_lnrbankloan) & !missing(x_lnrbankmort) & !missing(x_lnrbanknonmort)
gen d4x_bank_w1_balanced = 1 if !missing(d4x_lnrbankloan_w1) & !missing(d4x_lnrbankmort_w1) & !missing(d4x_lnrbanknonmort_w1)
local bankloan_list x_lnrbankloan x_lnrbankmort x_lnrbanknonmort
foreach v of local bankloan_list {
	gen `v'_b = `v' if x_bank_balanced == 1
	gen d4`v'_b_w1 = d4`v'_w1 if d4x_bank_w1_balanced == 1	
}

// Share data
gen x_share_m_hhdebt = 100*(exp(w_lnrmortdebt)*cpi)/hhdebt
gen x_share_b_hhdebt  = 100*x_bankloan/hhdebt
gen x_share_bm_bloan  = 100*x_bankmort/x_bankloan
label var x_share_m_hhdebt "Mortgage loans in percent of household debt"
label var x_share_b_hhdebt "Bank loans in percent of household debt"	
label var x_share_bm_bloan "Bank mortgage loans in percent of household loans by banks"

local share_varlist x_share_m_hhdebt x_share_b_hhdebt x_share_bm_bloan
foreach v of local share_varlist {
	replace `v' = . if `v' < 0 | `v' > 100
	local lab_tmp : variable label `v'
    qui gen d4`v' = `v' - L4.`v'
	label variable d4`v' "d4 `lab_tmp'"
	
	// Winsorize
	winsor d4`v', gen(d4`v'_w1) p(0.01)	
}


// Inflation
capture drop d1lncpi
capture drop d4lncpi
gen d1lncpi = 100*(lncpi-l1.lncpi)
gen d4lncpi = 100*(lncpi-l4.lncpi)
di d1lncpi "Inflation (Q-on-Q % change: 100*(lncpi-l1.lncpi)"
di d4lncpi "Inflation (Y-on-Y % change: 100*(lncpi-l4.lncpi)"

// Winsorize
capture drop d4lncpi_w1
winsor d4lncpi, gen(d4lncpi_w1) p(0.01)
label var d4lncpi_w1 "d4lncpi: winsorized at 0.01"	

// Fix creditgap_pos dummies for slope regressions
replace creditgap_pos = 0 if missing(creditgap_pos)
capture drop EM_creditgap_pos
gen EM_creditgap_pos = EM*creditgap_pos

*===============================================================================
* Estimation
*===============================================================================
//[Outcome variable] Note: d4 and _w1 will be automatically added below, e.g., d4lnrhhdebt_w1 d4lnrpvcon_w1
/*
local y_list lnrhhdebt lnrpvcon lnrgdp lnrp_house lncpi ///
	x_lnrbankmort x_lnrhhdebtexbm ///
	x_lnrbankloan x_lnrnonbankloan x_lnrbanknonmort ///
	w_lnrmortdebt w_lnrnonmortdebt ///
	x_lnrbankloan_b x_lnrbankmort_b x_lnrbanknonmort_b 
    *x_share_b_hhdebt x_share_bm_bloan x_share_m_hhdebt 
	*/
local y_list lnrhhdebt lnrpvcon
	
// Specify the list of country groups
local condlist ALL AE EM ASIA EUROPE AMERICAS HI_fx_regime_avg HI_kaopen_norm_avg HI_fin_dev_avg HIdtiQ1Q2 creditgap_pos EM_creditgap_pos ltv_level_high

local app_or_rep replace

local var2 dltv_level
if `flg_lnLTV' local var2 dlnltv_level

local addStatGMM addstat(AR(1) Test, e(ar1p), AR(2) Test, e(ar2p))
local addStatAREG addstat(Country no, e(N_clust))

// FE (OLS)
if `run_subsample_OLS'==1 {
	foreach y of local y_list {
		local var1 d4`y'_w1
		// Output excel file name
		local myfile="../Results/Macropru/Subsample/OLS/Subsample_OLS_`y'_`Spec'_`today'`Controls'`LnLTV'`NonIntro'`NonCRE'_v`str_data_version'.xls"
		local app_or_rep replace
		
		foreach var3 of local condlist {		
			local OtherReg l.strate l.d4lnrgdp_w1
			local regressors l.`var1' `OtherReg' l.`var2' l2.`var2' l3.`var2' l4.`var2'
			
			// FE (OLS) - subsample
			capture areg `var1' `regressors' i.dateq if `touse' & `var3'==1, absorb(ifscode) clu(ifscode)
			if _rc continue
			
			// Cumulative effect 		
			lincom l.`var2' + l2.`var2' + l3.`var2' +l4.`var2' 
			local m = r(estimate)
			local n = r(p)
			
			// Export results to the excel file
			outreg2  using `myfile', excel `app_or_rep' keep(`regressors') `addStatAREG' addtext(Errors, robust, TIME FE, YES, COU FE, YES, Period, >=`StartYear', Method, OLS, Group, `var3', Sum coeff ltv level, `m', P-value joint test, `n') label
			if "`app_or_rep'" == "replace" local app_or_rep append
		}
	}
}

// GMM (xtabond2)
if `run_subsample_GMM'==1 {
	xi i.dateq	
	foreach y of local y_list {
		local var1 d4`y'_w1
		// Output excel file name
		local myfile="../Results/Macropru/Subsample/GMM/Subsample_GMM_`y'_`Spec'_`today'`Controls'`LnLTV'`NonIntro'`NonCRE'_v`str_data_version'.xls"
		local app_or_rep replace
		
		foreach var3 of local condlist {		
			local OtherReg l.strate l.d4lnrgdp_w1
			local regressors l.`var1' `OtherReg' l.`var2' l2.`var2' l3.`var2' l4.`var2'
			
			// GMM (xtabond2) - subsample
			capture xtabond2 `var1' `regressors' _Idateq* if `touse' & `var3'==1, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
			if _rc continue
			
			// Cumulative effect 		
			lincom l.`var2' + l2.`var2' + l3.`var2' +l4.`var2' 
			local m = r(estimate)
			local n = r(p)
			
			// Export results to the excel file		
			outreg2  using `myfile', excel `app_or_rep' keep(`regressors') `addStatGMM' addtext(Errors, robust, TIME FE, YES, COU FE, YES, Period, >=`StartYear', Method, GMM, Group, `var3', Sum coeff ltv level, `m', P-value joint test, `n') label
			if "`app_or_rep'" == "replace" local app_or_rep append
		}
	}
}

// Slope dummies
if `run_slope_dummies'==1 {
	foreach y of local y_list {
		// Output excel file name
		local myfile="../Results/Macropru/Subsample/Slope/Slope_`Spec'_`y'_`today'`Controls'`LnLTV'`NonIntro'`NonCRE'_v`str_data_version'.xls"
		local app_or_rep replace
		
		foreach var3 of local condlist {
			// FE with interaction terms between initial high LTV level dummy and `dltv'	
			capture xi: areg d4`y'_w1 l.d4`y'_w1 l.strate l.d4lnrgdp_w1 `controls_FE' c.l.`var2'##i.`var3' c.l2.`var2'##i.`var3' c.l3.`var2'##i.`var3' c.l4.`var2'##i.`var3' `var3' i.dateq if `touse', absorb(ifscode) clu(ifscode)
			if _rc continue
			
			// Cumulative effect for the case of low initial LTV levels (lag)		
			capture lincom l.`var2' + l2.`var2' + l3.`var2' +l4.`var2' 
			if !_rc {
				local lincom_p_low   = r(p)
				local lincom_est_low = r(estimate)
			}
			else {
				local lincom_p_low   = .
				local lincom_est_low = .
			}
			
					
			// Cumulative effect for the case of high initial LTV levels (lag)
			capture lincom l.`var2' + l2.`var2' + l3.`var2' +l4.`var2' ///
				+ 1.`var3'#cL1.`var2' + 1.`var3'#cL2.`var2' ///
				+ 1.`var3'#cL3.`var2' + 1.`var3'#cL4.`var2'	
			if !_rc {
				local lincom_p_high   = r(p)
				local lincom_est_high = r(estimate)
			}
			else {
				local lincom_p_high   = .
				local lincom_est_high = .
			}
			
			// Difference between high and low initial LTV levels (lag)
			capture lincom 1.`var3'#cL1.`var2' + 1.`var3'#cL2.`var2' ///
				+ 1.`var3'#cL3.`var2' + 1.`var3'#cL4.`var2'	
			if !_rc {
				local lincom_p_diff   = r(p)
				local lincom_est_diff = r(estimate)
			}
			else {
				local lincom_p_diff   = .
				local lincom_est_diff = .
			}
							
			// Export results to the excel file
			outreg2  using `myfile', excel `app_or_rep' keep(l.d4`y'_w1 l.strate l.d4lnrgdp_w1 `controls_FE' `var3') addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, OLS, Period, >=`StartYear', Group for dummy, `var3', Cum eff if dummy 0, `lincom_est_low', P-value if dummy 0, `lincom_p_low', Cum eff if dummy 1, `lincom_est_high', P-value if dummy 1, `lincom_p_high', Diff, `lincom_est_diff', P-value Diff, `lincom_p_diff') label		
// 			outreg2  using `myfile', excel `app_or_rep' keep(l.d4`y'_w1 l.strate l.d4lnrgdp_w1 `controls_FE' c.l.`var2'##i.`var3' c.l2.`var2'##i.`var3' c.l3.`var2'##i.`var3' c.l4.`var2'##i.`var3' `var3') addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, OLS, Period, >=`StartYear', Group, ALL, Cum eff if dummy 0, `lincom_est_low', P-value if dummy 0, `lincom_p_low', Cum eff if dummy 1, `lincom_est_high', P-value if dummy 1, `lincom_p_high', Diff, `lincom_est_diff', P-value Diff, `lincom_p_diff') label		
			if "`app_or_rep'" == "replace" local app_or_rep append
			
		}
	}
}

**********************************
log close
exit


/*
//////////////////////////////////////////////////////// 
// Excerpt from the macropru_server_wp4_aa_rob-v2.do ///
//////////////////////////////////////////////////////// 

//////// lines 539-691 of macropru_server_wp4_aa_rob-v2.do //////////

***** LTV level - GMM - ****************
xtset ifscode quarter
sort ifscode quarter
cd "C:\Users\AAlter\Box Sync\iMaPP WP\Results\Macropru"
local addStatGMM addstat(AR(1) Test, e(ar1p), AR(2) Test, e(ar2p))
local addStatAREG addstat(Country no, e(N_clust))

local varlist dltv_level 
*demand_side_yoy supply_side_all_yoy supply_general_yoy supply_capital_yoy supply_loans_yoy allsum_yoy allloansum_yoy  dsti_yoy ltv_yoy lcg_yoy llp_yoy loanr_yoy tax_yoy
*loanr_yoy lcg_yoy lfc_yoy lfx_yoy llp_yoy lr_yoy ltd_yoy lvr_yoy capital_yoy conservation_yoy rr_yoy sifi_yoy tax_yoy ccb_yoy ot_yoy
*local deplist d4lnrpvdebt_w1 d4lnrhhdebt_w1 d4lnrcorpdebt_w1 d4lnrp_house_w1 d4lnrpvcon_w1
local deplist d4lnrhhdebt_w1 d4lnrpvcon_w1
*local condlist ALL AE EM ASIA EUROPE AMERICAS HI_fx_regime_avg HI_kaopen_norm_avg HI_financial_openness HI_fin_dev_avg HIdebtshare HIdtiQ1Q2 bis_creditgap_pos hprice_gap_pos
local condlist ALL AE EM ASIA EUROPE AMERICAS HI_fx_regime_avg HI_kaopen_norm_avg HI_fin_dev_avg HIdtiQ1Q2 creditgap_pos EM_creditgap_pos

  
foreach var1 in `deplist' {
local myfile="Results_macropru_wp0424_`var1'_gmm.xls"
	foreach var2 in `varlist' {
		foreach var3 in `condlist' {
		
		local suf=substr("`var1'",1,2)
		local OtherReg l.strate l.d4lnrgdp_w1
		
		local regressors l.`var1' `OtherReg' l.`var2' l2.`var2' l3.`var2' l4.`var2'
		
		***
		*qui areg `var1' `regressors' i.quarter if year>1999 &, absorb(ifscode) clu(ifscode)
		qui xtabond2 `var1' `regressors' _Iquarter* if year>1999 & `var3'==1, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
		
		local m=_b[l.`var2']+_b[l2.`var2']+_b[l3.`var2']+_b[l4.`var2']
		qui test l.dltv_level + l2.dltv_level + l3.dltv_level +l4.dltv_level = 0
		local n=r(p)
		outreg2  using `myfile', excel append keep(`regressors') `addStatGMM' addtext(Errors, robust, TIME FE, YES, COU FE, YES, Period, >1999, Method, GMM, Group, `var3', Sum coeff ltv level, `m', P-value joint test, `n')

		
		
		***GMM****

		*qui xtabond2 `var1' `regressors' _Iquarter* if year>1990, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
		*outreg2  using `myfile', excel append keep(`regressors') `addStatGMM' addtext(Errors, robust, COU FE, YES, TIME FE, YES, Method, GMM, Period, >1990, Group, ALL) label
		
		*qui xtabond2 `var1' `regressors' _Iquarter*  if year>1990 & AE==1, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
		*outreg2  using `myfile', excel append keep(`regressors') `addStatGMM' addtext(Errors, robust, COU FE, YES, TIME FE, YES, Method, GMM, Period, >1990, Group, AE) label
		
		*qui xtabond2 `var1' `regressors' _Iquarter*  if year>1990 & EM==1, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
		*outreg2  using `myfile', excel append keep(`regressors') `addStatGMM' addtext(Errors, robust, COU FE, YES, TIME FE, YES, Method, GMM, Period, >1990, Group, EM) label
		}
	}
	
}


*********************************************************************************
xtset ifscode quarter
drop if year<1989
tsfilter bk lnrhhd_bk = lnrhhd
tsfilter bw lnrhhd_bw = lnrhhd
tsfilter cf lnrhhd_cf = lnrhhd
tsfilter hp lnrhhd_hp = lnrhhd
tsfilter cf lnrhhd_cf4_40 = lnrhhd, minperiod(4) maxperiod(80) // 4 to 80 quarters instead of the default 1.5 to 8 years
tsfilter cf output_r_cf4_40 = lnrgdp, minperiod(6) maxperiod(32) // BCS with the default 1.5 to 8 years

*tsfilter cf lnrhhd_cf4_40_em = lnrhhd, minperiod(4) maxperiod(80) //this is for EMs only
*replace lnrhhd_cf4_40=lnrhhd_cf4_40_em if EM==1
*drop lnrhhd_cf4_40_em

*Adding a constant (k=100) as otherwise cyclical component is negative and will create issues for ln and d_ln calculations
replace lnrhhd_bk = lnrhhd_bk+100 if lnrhhd_bk!=.
replace lnrhhd_bw = lnrhhd_bw+100 if lnrhhd_bw!=.
replace lnrhhd_cf = lnrhhd_cf+100 if lnrhhd_cf!=.
replace lnrhhd_hp = lnrhhd_hp+100 if lnrhhd_hp!=.
replace lnrhhd_cf4_40 = lnrhhd_cf4_40+100 if lnrhhd_cf4_40!=.

*house price gap using CF-1-10 years
g lnrhhd_gap_cf4_40=(lnrhhd_cf4_40-100)/lnrhhd
label variable lnrhhd_gap_cf4_40 "cyclical component of house prices as a ratio of house prices [based on CF filter with 1-10 years])"

g output_gap_cf4_40=(output_r_cf4_40)/lnrgdp
label variable output_gap_cf4_40 "cyclical component of the real output as a ratio of real output [based on CF filter default parameters])"

gen pos_hhdgap_4q=0
replace pos_hhdgap_4q=1 if l.lnrhhd_gap_cf4_40>0 & l2.lnrhhd_gap_cf4_40>0 & l3.lnrhhd_gap_cf4_40>0 & l4.lnrhhd_gap_cf4_40>0 &!missing(l4.lnrhhd_gap_cf4_40)
gen pos_hhdgap_3q=0
replace pos_hhdgap_3q=1 if l.lnrhhd_gap_cf4_40>0 & l2.lnrhhd_gap_cf4_40>0 & l3.lnrhhd_gap_cf4_40>0 &!missing(l3.lnrhhd_gap_cf4_40)
gen pos_hhdgap_2q=0
replace pos_hhdgap_2q=1 if l.lnrhhd_gap_cf4_40>0 & l2.lnrhhd_gap_cf4_40>0 &!missing(l2.lnrhhd_gap_cf4_40)

gen neg_hhdgap_4q=0
replace neg_hhdgap_4q=1 if l.lnrhhd_gap_cf4_40<0 & l2.lnrhhd_gap_cf4_40<0 & l3.lnrhhd_gap_cf4_40<0 & l4.lnrhhd_gap_cf4_40<0
gen neg_hhdgap_3q=0
replace neg_hhdgap_3q=1 if l.lnrhhd_gap_cf4_40<0 & l2.lnrhhd_gap_cf4_40<0 & l3.lnrhhd_gap_cf4_40<0
gen neg_hhdgap_2q=0
replace neg_hhdgap_2q=1 if l.lnrhhd_gap_cf4_40<0 & l2.lnrhhd_gap_cf4_40<0

save "\\data2\gfsr\GFSR October 2017\GS\Household Debt\Analysis\Adrian\Macropru\quarterly_wp_macropru_final1.dta", replace

*********************************Credit Gap - iMaPP*******************************************

cd "C:\Users\AAlter\Box Sync\iMaPP WP\Data\Results\Macropru"
use "\\data2\gfsr\GFSR October 2017\GS\Household Debt\Analysis\Adrian\Macropru\quarterly_wp_macropru_final2.dta",replace
xtset ifscode quarter
sort ifscode quarter
local addStatGMM addstat(AR(1) Test, e(ar1p), AR(2) Test, e(ar2p))
local varlist demand_side_yoy supply_side_all_yoy supply_general_yoy supply_capital_yoy supply_loans_yoy allsum_yoy allloansum_yoy  dsti_yoy ltv_yoy lcg_yoy llp_yoy loanr_yoy tax_yoy
local deplist d4lnrhhdebt_w1 d4lnrpvcon_w1


foreach var1 in `deplist' {
local myfile="Results_macropru_wp0426_after2000_pos_gap_3q_`var1'.xls"
	foreach var2 in `varlist' {
		local suf=substr("`var1'",1,2)
		local OtherReg l.strate l.d4lnrgdp_w1
		local regressors l.`var1' `OtherReg' l.`var2' 
		
	
		qui xtreg `var1' `regressors' i.quarter if year>2000 ,fe clu(ifscode)
		outreg2  using `myfile', excel append keep(`regressors') addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, Xtreg, Period, >2000, Group, ALL) label
		***

		qui xtreg `var1' `regressors' i.quarter  if year>2000 & pos_hhdgap_3q==1 ,fe clu(ifscode)
		outreg2  using `myfile', excel append keep(`regressors') addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, Xtreg, Period, >2000, Group, Pos Credit Gap) label
		***

		qui xtreg `var1' `regressors' i.quarter  if year>2000 & neg_hhdgap_3q==1 ,fe clu(ifscode)
		outreg2  using `myfile', excel append keep(`regressors') addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, Xtreg, Period, >2000, Group, Neg Credit Gap) label
		
		****Conditional on Credit Gap and tightening/loosening
		
		qui xtreg `var1' `regressors' i.quarter  if year>2000 & pos_hhdgap_3q==1 & l.`var2'>=0 ,fe clu(ifscode)
		outreg2  using `myfile', excel append keep(`regressors') addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, Xtreg, Period, >2000, Group, PosCreditGap&Tightening) label
		***

		qui xtreg `var1' `regressors' i.quarter  if year>2000 & neg_hhdgap_3q==1 & l.`var2'<=0 ,fe clu(ifscode)
		outreg2  using `myfile', excel append keep(`regressors') addtext(Errors, robust, TIME FE, YES, COU FE, YES, Method, Xtreg, Period, >2000, Group, NegCreditGap&Loosening) label		
		
		
		
		/*
		***GMM****

		qui xtabond2 `var1' `regressors' _Iquarter* if year>2000, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
		outreg2  using `myfile', excel append keep(`regressors') `addStatGMM' addtext(Errors, robust, COU FE, YES, TIME FE, YES, Method, GMM, Period, >2000, Group, ALL) label
		
		qui xtabond2 `var1' `regressors' _Iquarter*  if year>2000 & AE==1, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
		outreg2  using `myfile', excel append keep(`regressors') `addStatGMM' addtext(Errors, robust, COU FE, YES, TIME FE, YES, Method, GMM, Period, >2000, Group, AE) label
		
		qui xtabond2 `var1' `regressors' _Iquarter*  if year>2000 & EM==1, gmm(L.(`var1' `regressors'), lag(1 1)) robust orthogonal
		outreg2  using `myfile', excel append keep(`regressors') `addStatGMM' addtext(Errors, robust, COU FE, YES, TIME FE, YES, Method, GMM, Period, >2000, Group, EM) label
		*/
	}
	
}
//////// lines 539-691 of macropru_server_wp4_aa_rob-v2.do //////////


