/*

IPW_estimation.do

Notes: This is the codes for the AIPW estimation for Section IV of the following paper.

	   Zohair Alam, Adrian Alter, Jesse Eiseman, Gaston Gelos, Heedon Kang, Machiko Narita, Erlend Nier, and Naixi Wang
	   "Digging Deeper – Evidence on the Effects of Macroprudential Policies from a New Database"   
	   
Last Updated: April 24, 2023 (Reflecting JMBC referee's comments)
	- Add house price and other MaPP as controls) in AIPW & FE
	- Streamlined the name of the output files, by dropping the elements that have been settled (e.g., Lag0, ALL, LTV_Qend). 
	  See the notes on Apr 23, 2022 of Notes.docx.
	- Datasets:
		- quarterly_wp_macropru_final5.dta: Data with the original iMaPP (WP version)
		- quarterly_wp_macropru_final6.dta: Data with the 2019 iMaPP		
		- quarterly_wp_macropru_final61.dta: Data with the 2019 iMaPP with Adrian's mortgage data
		- quarterly_wp_macropru_final7.dta: Data with the 2020 iMaPP with Adrian's mortgage data (used in Sep 5, 2021)
		- quarterly_wp_macropru_final8_iMaPP2019.dta: Data with the 2019 iMaPP with Adrian's mortgage data & comprehensive mortgage data
	
Last Updated: August 29, 2022
Previous version: IPW_estimation - 20220829 (before excl narrow changes).do

*/

*===============================================================================
* Step 1: Set up
*===============================================================================
//cd "C:\Users\mnarita\Box Sync\iMaPP WP\Codes"	
clear all 
set maxvar 10000
set more off
set excelxlsxlargefile on
pause off
capture log close

// Number of bins
local bin  "5bin" // "5bin" "4bin" "3bin"

// LTV variable to be used
local LTV_Qvar LTV_Qend // LTV_Qmax, LTV_Qmean, or LTV_Qend LTV_Qend_med

// Spec specification (two T groups, one L group)
local group_T
local group_L 
if "`bin'" == "4bin" | "`bin'" == "5bin" {	
	local group_T 10
	if "`bin'" == "5bin" {
		local group_L 10
	}
}
local extreme_T 25
local extreme_L 25
local inclusive = 0  // 0 to set "-`group_T' < dltv_level", 1 to set "-`group_T' <= dltv_level"

local flg_ctrl_lag_house_prices = 1 // Set 1 if the lag of house prices is included. (default is one)
local flg_ctrl_other_policies   = 1 // Set 1 if other MaPP policies (excl. LTV) is included. (default is one)

local flg_FE_lags_on = 1 // Set 1 if all four lags are included. (default is one)
local flg_nonIntro   = 0 // Set 1 to exclude obs with L_LTV >= 100, i.e., introduction of LTV. (default is zero)

// For T20 group, introduction cases account for 9 out of the 13 cases where bank loans etc. available.
// Among the remaining 4, two of them (Malaysia 2011Q4, Poland 2014Q2) are found to be LTV limits on loans for "commercial residential estates".
// To drop these cases for all estimations, use the following flag variable (default is one):
local flg_nonCRE = 1 // Set 1 to exclude CRE LTV tightening (Hong Kong 1994Q1 and 2013Q1, Lebanon 2008Q3, Malaysia 2011Q4, Poland 2014Q2).

// For T20 group, there are four cases where changes seem to happen only on relatively narrow real estate markets (although we cannot be sure without detailed market share data).
// To drop these cases for all estimations, use the following flag variable (default is zero):
local flg_nonNarrow = 1 // Set 1 to exclude LTV tightening with a relateively narrow scope (Denmank 2003Q2, Malaysia 2010Q4, Singapore 2012Q4 and 2013Q1).

// Sample restrictions
local Country   ALL // ALL EM AE
local StartYear 1990 // 1990 2000
local SmplPeriod      "year >= `StartYear' & year <= 2016"
local runWithDataAfter = 1  // Data to use (0: Before or 1: after revision)
// local str_data_version 8_iMaPP2019	// e.g., set to "8_iMaPP2019" to use "quarterly_wp_macropru_final8_iMaPP2019.dta"
local str_data_version 8_iMaPP2021	//61 // e.g., set to "8_iMaPP2021" to use "quarterly_wp_macropru_final8_iMaPP2021.dta"
if `runWithDataAfter'== 0 local str_data_version 5	// The "before" data file is "quarterly_wp_macropru_final5.dta"

// Spec name: 
// Example: T1025_L10_in0_2000
// local group_T 10
// local group_L 10
// local extreme_T 25
// local extreme_L 
// local StartYear 2000 
// local inclusive = 0
local Spec "T`group_T'`extreme_T'_L`group_L'`extreme_L'_in`inclusive'_`StartYear'" 
di "Spec = `Spec'"

// Other controls
local controls_TM
local controls_OM
local controls_FE
local Controls ""
if `flg_ctrl_lag_house_prices'==1 {	
	// Includ house prices as a control
	local controls_TM `controls_TM' l.d4lnrp_house_w1
	local controls_OM `controls_OM' d4lnrp_house_w1
	local controls_FE `controls_FE' l.d4lnrp_house_w1
	local Controls `Controls'_HP
}
if `flg_ctrl_other_policies'==1 {	
	// Includ other MaPP policies (excl. LTV) as a control: Average of the dummies from t to t-3.
	local controls_TM `controls_TM' l.allsum_wo_ltv_yoy
	local controls_OM `controls_OM' allsum_wo_ltv_yoy
 	*local controls_FE `controls_FE' l.allsum_wo_ltv  l2.allsum_wo_ltv  l3.allsum_wo_ltv  l4.allsum_wo_ltv
 	*local Controls `Controls'_OMaPP4
	local controls_FE `controls_FE' l.allsum_wo_ltv_yoy 
	local Controls `Controls'_OMaPP
}
if `flg_FE_lags_on'==0 local controls_FE `controls_OM' // For comparison purposes


local NonIntro ""
if `flg_nonIntro'==1 {
	local NonIntro "_NonIntro"
}

local NonCRE ""
if `flg_nonCRE'==1 {
	local NonCRE "_NonCRE"
}

local NonNarrow ""
if `flg_nonNarrow'==1 {
	local NonNarrow "_NonNarrow"
}

local FElagon ""
if `flg_FE_lags_on'== 0 local FElagon "_FElagon0"


// Local variable that summarizes sample to be used:
local touse `SmplPeriod' 
di "touse = `touse'"

// Set the log file
local today : display %tdCYND date(c(current_date), "DMY")  // Date (YYYYMMDD)
log using "../Results/Macropru/AIPW_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'`FElagon'_v`str_data_version'.txt", text replace

di "Spec = `Spec'"
di "touse = `touse'"

local TM_estimation 1 // 1: Estimate the treatment model, 0: Skip

// dltv_level: winsorized or not
local dltv_var  dltv_level  //dltv_level dltv_level_w1

*-------------------------------------------------------------------------------
* Load data
*-------------------------------------------------------------------------------
local ltv_related_var_list `dltv_var' 
if `runWithDataAfter' == 1 {
	*use "../Data/quarterly_wp_macropru_final6.dta", clear // iMaPP-2019
// 	use "../Data/quarterly_wp_macropru_final61.dta", clear
	use "../Data/quarterly_wp_macropru_final`str_data_version'.dta", clear
}
else {
	//  use "\\data1\MCM\DRAFTS\MP\iMaPP Database\2 iMaPP - Master\2019 update\2 Update\3 Checks\3 Analysis\Data\quarterly_wp_macropru_final6.dta", clear
	* use "C:\Users\MNarita\Box Sync\iMaPP WP\Data\quarterly_wp_macropru_final5.dta", clear
	use "../Data/quarterly_wp_macropru_final`str_data_version'.dta", clear
}

*-------------------------------------------------------------------------------
* Time variables
*-------------------------------------------------------------------------------
*rename  quarter dateq
capture gen qr = quarter(dofq(dateq))
label var dateq "Time variable in Stata format (quarterly:%tq)"
label var qr    "Quarter"
xtset isocode dateq

// For comparison, drop observations in the first quarter
if `runWithDataAfter' {
	foreach v of local ltv_related_var_list {
		replace `v' = . if dateq <= yq(`StartYear',1)
	}
}
capture drop _Iquarter_*

// Keep the selected sample only
di as err "Sample: `Country' +++++++++++ (cmd:keep if `Country' == 1) +++++++++++" 
su ALL AE EM
keep if `Country'   == 1
su ALL AE EM if ALL == 1 
su ALL AE EM if AE  == 1 
su ALL AE EM if EM  == 1 

*-------------------------------------------------------------------------------
* dltv_level variables
*-------------------------------------------------------------------------------
// dltv_level variable to use
capture drop ltv_level 
capture drop dltv_level
gen 	  ltv_level  = `LTV_Qvar' 
gen       dltv_level = d.ltv_level
label var ltv_level  "`LTV_Qvar': Quarterly aggregation of LTV_average"
label var dltv_level "d.`LTV_Qvar': Quarterly aggregation of LTV_average"

// LTV level in the previous quarter
capture drop L_LTV
gen L_LTV = l.ltv_level
label var L_LTV "Lag of ltv_level"

// Dummy variable to indicate an introduction of LTV limits (used to exclude these cases)
// For Malaysia and Romania, there are two incidences of LTV introductions 
// because the limits were once abolished (intead of being set at 100) but re-introduced later.
egen ltv_max_by_country = max(ltv_level), by(ifscode)
gen byte dltv_intro = (L_LTV == ltv_max_by_country & 100 <= ltv_max_by_country & dltv_level < 0)
label var dltv_intro "Dummy variable: 1 indicates an introduction of LTV limits"

if `flg_nonIntro'==1 {
	replace ltv_level  = . if dltv_intro == 1
	replace dltv_level = . if dltv_intro == 1
}
if `flg_nonCRE'==1 {
	// Exclude CRE LTV tightening (Hong Kong 1994Q1 and 2013Q1, Lebanon 2008Q3, Malaysia 2011Q4, Poland 2014Q2).
	replace ltv_level  = . if iso3 == "HKG" & dateq == yq(1994,1)
	replace ltv_level  = . if iso3 == "HKG" & dateq == yq(2013,1)
 	*replace ltv_level  = . if iso3 == "LBN" & dateq == yq(2008,3) // Not in the estimation sample
	replace ltv_level  = . if iso3 == "MYS" & dateq == yq(2011,4)
	replace ltv_level  = . if iso3 == "POL" & dateq == yq(2014,2)
	
	replace dltv_level = . if iso3 == "HKG" & dateq == yq(1994,1)
	replace dltv_level = . if iso3 == "HKG" & dateq == yq(2013,1)
 	*replace dltv_level = . if iso3 == "LBN" & dateq == yq(2008,3) // Not in the estimation sample
	replace dltv_level = . if iso3 == "MYS" & dateq == yq(2011,4)
	replace dltv_level = . if iso3 == "POL" & dateq == yq(2014,2)
}

if `flg_nonNarrow'==1 {
	// Exclude LTV tightening with a relateively narrow scope (Denmank 2003Q2, Malaysia 2010Q4, Singapore 2012Q4 and 2013Q1).
	replace ltv_level  = . if iso3 == "DNK" & dateq == yq(2003,2)
	replace ltv_level  = . if iso3 == "MYS" & dateq == yq(2010,4)
	replace ltv_level  = . if iso3 == "SGP" & dateq == yq(2012,4)
	replace ltv_level  = . if iso3 == "SGP" & dateq == yq(2013,1)
	
	replace dltv_level = . if iso3 == "DNK" & dateq == yq(2003,2)
	replace dltv_level = . if iso3 == "MYS" & dateq == yq(2010,4)
	replace dltv_level = . if iso3 == "SGP" & dateq == yq(2012,4)
	replace dltv_level = . if iso3 == "SGP" & dateq == yq(2013,1)
}

// Trim outliers if specified
if "`extreme_T'" != "" {
	replace ltv_level  = . if -`extreme_T' > dltv_level // "-25 > dltv_level" to exclude very large tightening
	replace dltv_level = . if -`extreme_T' > dltv_level // "-25 > dltv_level" to exclude very large tightening	
}
if "`extreme_L'" != "" {
	replace ltv_level  = . if dltv_level > `extreme_L' // "dltv_level > 25" to exclude very large loosening
	replace dltv_level = . if dltv_level > `extreme_L' // "dltv_level > 25" to exclude very large loosening	
}

// Winsorize dltv_level
capture drop   dltv_level_w1*
winsor    dltv_level if dltv_level ~=0 , gen(dltv_level_w1) p(0.1)
replace   dltv_level_w1 = 0 if dltv_level ==0
label var dltv_level_w1 "dltv_level_w, Winsorized fraction .1"

/*
// Summary table to a Word file
xtsum2docx ltv_level if ltv_level<0 using test.docx, ///
 replace mean(%9.1f) p25(%9.1f) median(%9.1f) p75(%9.1f) obs(%9.0fc) xtn(%9.0fc) order(mean p25 median p75 obs xtn)
*/

*-------------------------------------------------------------------------------
* Other variables
*-------------------------------------------------------------------------------
// Rename the mortgage variables from Adrian to keep the order in the result file
rename lnrmortdebt         w_lnrmortdebt 
rename d4lnrmortdebt       d4w_lnrmortdebt
rename d4lnrmortdebt_w1    d4w_lnrmortdebt_w1 
replace d4w_lnrmortdebt    = 100*d4w_lnrmortdebt
replace d4w_lnrmortdebt_w1 = 100*d4w_lnrmortdebt_w1
label var w_lnrmortdebt   "log of mortgage debt by Adrian"
label var d4w_lnrmortdebt "Y-on-Y change (x-l4.x): x is w_lnrmortdebt by Adrian"	
label var d4w_lnrmortdebt_w1 "d4w_lnrmortdebt: winsorized at 0.01"	

rename lnrnonmortdebt         w_lnrnonmortdebt
rename d4lnrnonmortdebt       d4w_lnrnonmortdebt
rename d4lnrnonmortdebt_w1    d4w_lnrnonmortdebt_w1
replace d4w_lnrnonmortdebt    = 100*d4w_lnrnonmortdebt
replace d4w_lnrnonmortdebt_w1 = 100*d4w_lnrnonmortdebt_w1
label var w_lnrnonmortdebt   "log of non-mortgage debt by Adrian"
label var d4w_lnrnonmortdebt "Y-on-Y change (x-l4.x): x is w_lnrnonmortdebt by Adrian"	
label var d4w_lnrnonmortdebt_w1 "d4w_lnrnonmortdebt: winsorized at 0.01"	


// Balanced bank data series
gen x_bank_balanced = 1 if !missing(x_lnrbankloan) & !missing(x_lnrbankmort) & !missing(x_lnrbanknonmort)
gen d4x_bank_w1_balanced = 1 if !missing(d4x_lnrbankloan_w1) & !missing(d4x_lnrbankmort_w1) & !missing(d4x_lnrbanknonmort_w1)
local bankloan_list x_lnrbankloan x_lnrbankmort x_lnrbanknonmort
foreach v of local bankloan_list {
	gen `v'_b = `v' if x_bank_balanced == 1
	gen d4`v'_b_w1 = d4`v'_w1 if d4x_bank_w1_balanced == 1	
}

// Share data
gen x_share_m_hhdebt = 100*(exp(w_lnrmortdebt)*cpi)/hhdebt
gen x_share_b_hhdebt  = 100*x_bankloan/hhdebt
gen x_share_bm_bloan  = 100*x_bankmort/x_bankloan
label var x_share_m_hhdebt "Mortgage loans in percent of household debt"
label var x_share_b_hhdebt "Bank loans in percent of household debt"	
label var x_share_bm_bloan "Bank mortgage loans in percent of household loans by banks"

local share_varlist x_share_m_hhdebt x_share_b_hhdebt x_share_bm_bloan
foreach v of local share_varlist {
	replace `v' = . if `v' < 0 | `v' > 100
	local lab_tmp : variable label `v'
    qui gen d4`v' = `v' - L4.`v'
	label variable d4`v' "d4 `lab_tmp'"
	
	// Winsorize
	winsor d4`v', gen(d4`v'_w1) p(0.01)	
}


// Inflation
capture drop d1lncpi
capture drop d4lncpi
gen d1lncpi = 100*(lncpi-l1.lncpi)
gen d4lncpi = 100*(lncpi-l4.lncpi)
di d1lncpi "Inflation (Q-on-Q % change: 100*(lncpi-l1.lncpi)"
di d4lncpi "Inflation (Y-on-Y % change: 100*(lncpi-l4.lncpi)"

// Winsorize
capture drop d4lncpi_w1
winsor d4lncpi, gen(d4lncpi_w1) p(0.01)
label var d4lncpi_w1 "d4lncpi: winsorized at 0.01"	

*===============================================================================
* Treatment groups: dltv = -10 and 10 (Control: dltv == 0)
*===============================================================================
capture drop flg_dltv* dltv_level_*

if "`bin'" == "5bin" {
	local dltv_list -20 -10 10 20 // Potential outcome: descretized values of dltv

	// Construct the discretized variable of LTV limit changes 
	// Note: always be based on the original series (i.e., dltv_level)
	drop    dltv
	if `inclusive' == 1 {
		// Note: `inclusive'=1 will set "-`group_T' <= dltv_level"
		local ineq_T1  <=
		local ineq_T2  <
		local ineq_L1  <=
		local ineq_L2  <
	}
	else {
		// Note: `inclusive'=0 will set "-`group_T' < dltv_level"
		local ineq_T1  <
		local ineq_T2  <=
		local ineq_L1  <
		local ineq_L2  <=
	}
	gen     dltv = 0   if !missing(dltv_level)
	replace dltv = -10 if -`group_T' `ineq_T1' dltv_level  & dltv_level < 0 & !missing(dltv_level)
	replace dltv = -20 if dltv_level `ineq_T2' -`group_T' & !missing(dltv_level)
	replace dltv = 10  if 0 < dltv_level  & !missing(dltv_level)
	replace dltv = 10  if 0 < dltv_level & dltv_level `ineq_L1' `group_L' & !missing(dltv_level)
	replace dltv = 20  if `group_L' `ineq_L2' dltv_level  & !missing(dltv_level)
	label var dltv "Buckets of dltv_level"

	// Histogram of the descretized LTV changes
	// hist dltv if dltv_level != 0

	// Dummy variable for each group
	gen flg_dltv0 = (dltv == 0)   if !missing(dltv)
	gen flg_dltv1 = (dltv == -20) if !missing(dltv)
	gen flg_dltv2 = (dltv == -10) if !missing(dltv)
	gen flg_dltv3 = (dltv ==  10) if !missing(dltv)
	gen flg_dltv4 = (dltv ==  20) if !missing(dltv)
	label var flg_dltv0 "Dummy for (dltv == 0)" 
	label var flg_dltv1 "Dummy for (dltv == -20)" 
	label var flg_dltv2 "Dummy for (dltv == -10)"
	label var flg_dltv3 "Dummy for (dltv ==  10)"
	label var flg_dltv4 "Dummy for (dltv ==  20)"
	//br isocode dateq dltv flg_dltv*

	// dLTV variables for FE regression
	gen dltv_level_T10 = dltv_level*(-`group_T' `ineq_T1' dltv_level & dltv_level < 0) if !missing(dltv_level)
	gen dltv_level_T20 = dltv_level*(dltv_level `ineq_T2' -`group_T') if !missing(dltv_level)
	gen dltv_level_L10 = dltv_level*(0 < dltv_level & dltv_level `ineq_L1' `group_L') if !missing(dltv_level)
	gen dltv_level_L20 = dltv_level*(`group_L' `ineq_L2' dltv_level) if !missing(dltv_level)
	label var dltv_level_T10 "dltv_level: tightening: -`group_T' `ineq_T1' dltv_level"
	label var dltv_level_T20 "dltv_level: tightening: dltv_level `ineq_T2' -`group_T'"
	label var dltv_level_L10 "dltv_level: loosening: dltv_level `ineq_L1' `group_L'"
	label var dltv_level_L20 "dltv_level: loosening: `group_L' `ineq_L2' dltv_level"
}
if "`bin'" == "4bin" {
	local dltv_list -20 -10 10 // Potential outcome: descretized values of dltv

	// Construct the discretized variable of LTV limit changes 
	// Note: always be based on the original series (i.e., dltv_level)
	drop    dltv
	if `inclusive' == 1 {
		// Note: `inclusive'=1 will set "-`group_T' <= dltv_level"
		local ineq_T1  <=
		local ineq_T2  <
	}
	else {
		// Note: `inclusive'=0 will set "-`group_T' < dltv_level"
		local ineq_T1  <
		local ineq_T2  <=
	}
	gen     dltv = 0   if !missing(dltv_level)
	replace dltv = -10 if -`group_T' `ineq_T1' dltv_level  & dltv_level < 0 & !missing(dltv_level)
	replace dltv = -20 if dltv_level `ineq_T2' -`group_T' & !missing(dltv_level)
	replace dltv = 10  if 0 < dltv_level  & !missing(dltv_level)
	label var dltv "Buckets of dltv_level"

	// Histogram of the descretized LTV changes
	// hist dltv if dltv_level != 0

	// Dummy variable for each group
	gen flg_dltv0 = (dltv == 0)   if !missing(dltv)
	gen flg_dltv1 = (dltv == -20) if !missing(dltv)
	gen flg_dltv2 = (dltv == -10) if !missing(dltv)
	gen flg_dltv3 = (dltv ==  10) if !missing(dltv)
	label var flg_dltv0 "Dummy for (dltv == 0)" 
	label var flg_dltv1 "Dummy for (dltv == -20)" 
	label var flg_dltv2 "Dummy for (dltv == -10)"
	label var flg_dltv3 "Dummy for (dltv ==  10)"
	//br isocode dateq dltv flg_dltv*

	// dLTV variables for FE regression
	gen dltv_level_T10 = dltv_level*(-`group_T' `ineq_T1' dltv_level & dltv_level < 0) if !missing(dltv_level)
	gen dltv_level_T20 = dltv_level*(dltv_level `ineq_T2' -`group_T') if !missing(dltv_level)
	gen dltv_level_L10 = dltv_level*(0 < dltv_level) if !missing(dltv_level)
	label var dltv_level_T10 "dltv_level: tightening: -`group_T' `ineq_T1' dltv_level"
	label var dltv_level_T20 "dltv_level: tightening: dltv_level `ineq_T2' -`group_T'"
	label var dltv_level_L10 "dltv_level: loosening"
}
if "`bin'" == "3bin" {
	local dltv_list -10 10   // Potential outcome: descretized values of dltv

	// Construct the discretized variable of LTV limit changes
	drop dltv
	gen     dltv = 0   if !missing(dltv_level)
	replace dltv = -10 if dltv_level < 0   & !missing(dltv_level)
	replace dltv = 10  if 0 < dltv_level  & !missing(dltv_level)
	label var dltv "Buckets of dltv_level"
		
	//Histogram of the descretized LTV changes
	*hist dltv if dltv_level != 0

	// Dummy variable for each group
	gen flg_dltv0 = (dltv == 0)   if !missing(dltv)
	gen flg_dltv1 = (dltv == -10) if !missing(dltv)
	gen flg_dltv2 = (dltv ==  10) if !missing(dltv)
	label var flg_dltv0 "Dummy for (dltv == 0)" 
	label var flg_dltv1 "Dummy for (dltv == -10)"
	label var flg_dltv2 "Dummy for (dltv ==  10)"
	*br isocode dateq dltv flg_dltv*	

	// dLTV variables for FE regression
	gen dltv_level_T10 = dltv_level*(dltv_level < 0) if !missing(dltv_level)
	gen dltv_level_L10 = dltv_level*(0 < dltv_level) if !missing(dltv_level)
	label var dltv_level_T10 "dltv_level: tightening"
	label var dltv_level_L10 "dltv_level: loosening"
}
*===============================================================================
* `dltv_var' statistics
*===============================================================================
// Average change in LTV in each group
di "Looking at `bin': dltv_list is `dltv_list'"
local cnt 0
foreach dltv_value of local dltv_list { 
	local cnt = `cnt' + 1
	di "--------------------------------------------------"
	di "cnt `cnt': dltv is `dltv_value', using `dltv_var' "
	di "--------------------------------------------------"
	egen dltv`cnt'_a = mean(`dltv_var') if `touse' & dltv == `dltv_value'
	label var dltv`cnt'_a "Average change in LTV (`dltv_var') if dltv == `dltv_value'"
	
	// Average dltv
	su dltv`cnt'_a if dltv == `dltv_value'
	local dltv`cnt'_a_local = r(mean)
	di "dltv`cnt'_a_local = `dltv`cnt'_a_local'"
	local dltv`cnt'_obs2_local = r(N)
	di "dltv`cnt'_obs2_local = `dltv`cnt'_obs2_local'"
}
// Statistics (Table 3)
version 16: table dltv if `touse', c(count `dltv_var' mean `dltv_var' sd `dltv_var')
version 16: table dltv if AE == 1 & `touse' , c(count `dltv_var' mean `dltv_var' sd `dltv_var')
version 16: table dltv if EM == 1 & `touse' , c(count `dltv_var' mean `dltv_var' sd `dltv_var')

*===============================================================================
* Step 2: Treatment model of the change in the LTV limits
* Note: Ordered logit to obtain predicted probabilities (p0, p1, p2, p3) 
*===============================================================================
if `TM_estimation' == 1 {
	* Choose the type of the ordered logit with fixed effects
	* 1: ologit with ISO Dummies (This would not be consistent given T but might be fine when T is large)
	* 2: buclogit (DvS estimator Baetschmann et al. 2011, whose codes are provided by Dickerson, Hole, Munford (2014))
	* 3: ologit with Randome Effects
	local logit_type 1

	if `logit_type' == 1 {	
		*-------------------------------------------------------------------------------
		* 1: Ordered logit with Fixed Effects - using ISO Dummies
		* Note: This would not be consistent given T but might be fine when T is large
		*-------------------------------------------------------------------------------
		drop p1 p2 p3 p4 // Drop very old variables
		//di "cmd: ologit dltv l(1/2).d4lnrhhdebt_w1 l(1/2).d4lnrgdp_w1 l(1/2).strate l(1/2).vix i.qr i.isocode"
		di "cmd: ologit dltv l.d4lnrhhdebt_w1 l.d4lnrgdp_w1 l.strate `controls_TM' i.dateq i.isocode if `touse', cluster(isocode)"
		ologit dltv l.d4lnrhhdebt_w1 l.d4lnrgdp_w1 l.strate `controls_TM' i.dateq i.isocode ///		
		       if `touse', cluster(isocode)		

		if "`bin'" == "5bin" {
			di as err "Ordered logit: `bin'" 
			predict p1 p2 p3 p4 p5, pr
			label var p1 "Prob(dltv=-20)"
			label var p2 "Prob(dltv=-10)"
			label var p3 "Prob(dltv=0)"
			label var p4 "Prob(dltv=10)"
			label var p5 "Prob(dltv=20)"
			
			// For convenience in the following parts
			rename p3 p0
			label var p0 "Prob(dltv=0)"
			gen p0_1 = 1-p0 
			label var p0_1 "1-Prob(dltv=0)"
			rename p4 p3
			label var p3 "Prob(dltv=10)"
			rename p5 p4
			label var p4 "Prob(dltv=20)"
			
			// Truncate/winsorize propensity score (as in Jorda and Taylor, 2015)
			local cnt_list 0 1 2 3 4
			local p_cutoff = 0.1
			local one_minus_p_cutoff = 1 - `p_cutoff'
			foreach cnt of local cnt_list { 
				gen 	p`cnt'_w  = p`cnt'
				replace p`cnt'_w = `one_minus_p_cutoff' if p`cnt'_w > `one_minus_p_cutoff' & p`cnt'_w ~= .
				replace p`cnt'_w = `p_cutoff' if p`cnt'_w < `p_cutoff' & p`cnt'_w ~= .
				label var p`cnt'_w "Winsorized p`cnt' at `p_cutoff' and `one_minus_p_cutoff'" 
			}
		}
		if "`bin'" == "4bin" {		
			di as err "Ordered logit: `bin'" 
			predict p1 p2 p3 p4, pr
			label var p1 "Prob(dltv=-20)"
			label var p2 "Prob(dltv=-10)"
			label var p3 "Prob(dltv=0)"
			label var p4 "Prob(dltv=10)"
			
			// For convenience in the following parts
			rename p3 p0
			label var p0 "Prob(dltv=0)"
			gen p0_1 = 1-p0 
			label var p0_1 "1-Prob(dltv=0)"
			rename p4 p3
			label var p3 "Prob(dltv=10)"
			
			// Truncate/winsorize propensity score (as in Jorda and Taylor, 2015)
			local cnt_list 0 1 2 3
			local p_cutoff = 0.1
			local one_minus_p_cutoff = 1 - `p_cutoff'
			foreach cnt of local cnt_list { 
				gen 	p`cnt'_w  = p`cnt'
				replace p`cnt'_w = `one_minus_p_cutoff' if p`cnt'_w > `one_minus_p_cutoff' & p`cnt'_w ~= .
				replace p`cnt'_w = `p_cutoff' if p`cnt'_w < `p_cutoff' & p`cnt'_w ~= .
				label var p`cnt'_w "Winsorized p`cnt' at `p_cutoff' and `one_minus_p_cutoff'" 
			}
		}
		if "`bin'" == "3bin" {
			di as err "Ordered logit: `bin'" 
			predict p1 p2 p3, pr
			label var p1 "Prob(dltv=-10)"
			label var p2 "Prob(dltv=0)"
			label var p3 "Prob(dltv=10)"			
			
			// For convenience in the following parts
			rename    p2 p0
			label var p0 "Prob(dltv=0)"
			gen       p0_1 = 1-p0 
			label var p0_1 "1-Prob(dltv=0)"
			rename    p3 p2
			label var p2 "Prob(dltv=10)"
					
			// Truncate/winsorize propensity score (as in Jorda and Taylor, 2015)
			local cnt_list 0 1 2
			local p_cutoff = 0.1
			local one_minus_p_cutoff = 1 - `p_cutoff'
			foreach cnt of local cnt_list { 
				gen 	p`cnt'_w  = p`cnt'
				replace p`cnt'_w = `one_minus_p_cutoff' if p`cnt'_w > `one_minus_p_cutoff' & p`cnt'_w ~= .
				replace p`cnt'_w = `p_cutoff' if p`cnt'_w < `p_cutoff' & p`cnt'_w ~= .
				label var p`cnt'_w "Winsorized p`cnt' at `p_cutoff' and `one_minus_p_cutoff'" 
			}
		}		
	}	
	/*
	// Histograms of estimated probabilities
	twoway (histogram p0_1, start(0) fcolor(red) lcolor(red))|| ///
		   (histogram p1, start(0) fcolor(blue) lcolor(blue)) || ///
		   (histogram p2, start(0) fcolor(none) lcolor(green)) || ///
		   (histogram p3, start(0) fcolor(none) lcolor(black)), ///
		   legend(order(1 "1-Prob(dltv=0)" 2 "Prob(dltv=-20)" 3 "Prob(dltv=-10)" 4 "Prob(dltv=10)"))
	*/
	*-------------------------------------------------------------------------------
	* Propensity scores	
	*-------------------------------------------------------------------------------
	if "`bin'" == "3bin" {
		di as err "Histograms: `bin'" 		
		// Propensity score, not winsorized
		gen     PP = .
		replace PP = p1   if dltv == -10 
		replace PP = p0   if dltv == 0 
		replace PP = p2   if dltv == 10 
		label var PP  "Propensity score (p0,p1,p2): not winsorized"

		// Propensity score, winsorized at 0.1 and 0.9
		gen     PP_w = .
		replace PP_w = p1_w   if dltv == -10 
		replace PP_w = p0_w   if dltv == 0 
		replace PP_w = p2_w   if dltv == 10 
		label var PP_w  "Propensity score (p0,p1,p2): winsorized"

		// Probability of any treatment (1-p0) for the control group
		gen PP_1    = 1-PP      if dltv == 0 
		gen PP_1_w  = 1-PP_w    if dltv == 0 
		label var PP_1    "1-Prob(dltv=0); not winsorized"
		label var PP_1_w  "1-Prob(dltv=0); winsorized"

		// Figure: Estimated probability of Treatment (dltv=-10)
		twoway (histogram PP if dltv==-10, start(0) fcolor(red) lcolor(red) bin(15))  || ///
			   (histogram PP_1 if dltv==0, start(0) fcolor(none) lcolor(black)  bin(15)) ///
			   if `touse', ///
			   title("Estimated Probability of Treatment (`Country')") legend(order(1 "Treated (dltv=-10)" 2 "Control (dltv==0)"))
			   graph export "../Charts/Pr_Treat_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'(dltv=-10)_v`str_data_version'.emf", as(emf) replace	
		// Figure: Estimated probability of Treatment (dltv=10)
		twoway (histogram PP if dltv==10, start(0) fcolor(red) lcolor(red) bin(15))  || ///
			   (histogram PP_1 if dltv==0, start(0) fcolor(none) lcolor(black)  bin(15)) ///
			   if `touse', ///
			   title("Estimated Probability of Treatment (`Country')") legend(order(1 "Treated (dltv=10)" 2 "Control (dltv==0)"))
			   graph export "../Charts/Pr_Treat_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'(dltv=10)_v`str_data_version'.emf", as(emf) replace	
	
	}
	if "`bin'" == "4bin" | "`bin'" == "5bin" {	
		di as err "Histograms: `bin'" 	
		// Propensity score, not winsorized
		gen     PP = .
		replace PP = p1   if dltv == -20 
		replace PP = p2   if dltv == -10 
		replace PP = p0   if dltv == 0 
		replace PP = p3   if dltv == 10 
		if "`bin'" == "5bin" {	
			replace PP = p4   if dltv == 20 
		}
		label var PP  "Propensity score (p0,p1,p2,p3,p4): not winsorized"

		// Propensity score, winsorized at 0.1 and 0.9
		gen     PP_w = .
		replace PP_w = p1_w   if dltv == -20 
		replace PP_w = p2_w   if dltv == -10 
		replace PP_w = p0_w   if dltv == 0 
		replace PP_w = p3_w   if dltv == 10 
		if "`bin'" == "5bin" {	
			replace PP_w = p4_w   if dltv == 20 
		}
		label var PP_w  "Propensity score (p0,p1,p2,p3,p4): winsorized"

		// Probability of any treatment (1-p0) for the control group
		gen PP_1    = 1-PP      if dltv == 0 
		gen PP_1_w  = 1-PP_w    if dltv == 0 
		label var PP_1    "1-Prob(dltv=0); not winsorized"
		label var PP_1_w  "1-Prob(dltv=0); winsorized"
		
		// Figure: Estimated probability of Treatment (dltv=-10)
		twoway (histogram PP if dltv==-10, start(0) fcolor(red) lcolor(red) bin(15))  || ///
			   (histogram PP_1 if dltv==0, start(0) fcolor(none) lcolor(black)  bin(15)) ///
			   if `touse', ///
			   title("Estimated Probability of Treatment (`Country')") legend(order(1 "Treated (dltv=-10)" 2 "Control (dltv==0)"))
			   graph export "../Charts/Pr_Treat_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'(dltv=-10)_v`str_data_version'.emf", as(emf) replace
		// Figure: Estimated probability of Treatment (dltv=-20)
		twoway (histogram PP if dltv==-20, start(0) fcolor(red) lcolor(red) bin(15))  || ///
			   (histogram PP_1 if dltv==0, start(0) fcolor(none) lcolor(black)  bin(15)) ///
			   if `touse', ///
			   title("Estimated Probability of Treatment (`Country')") legend(order(1 "Treated (dltv=-20)" 2 "Control (dltv==0)"))
			   graph export "../Charts/Pr_Treat_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'(dltv=-20)_v`str_data_version'.emf", as(emf) replace
		// Figure: Estimated probability of Treatment (dltv=10)
		twoway (histogram PP if dltv==10, start(0) fcolor(red) lcolor(red) bin(15))  || ///
			   (histogram PP_1 if dltv==0, start(0) fcolor(none) lcolor(black)  bin(15)) ///
			   if `touse', ///
			   title("Estimated Probability of Treatment (`Country')") legend(order(1 "Treated (dltv=10)" 2 "Control (dltv==0)"))
			   graph export "../Charts/Pr_Treat_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'(dltv=10)_v`str_data_version'.emf", as(emf) replace
		if "`bin'" == "5bin" {	
			// Figure: Estimated probability of Treatment (dltv=20)
			twoway (histogram PP if dltv==20, start(0) fcolor(red) lcolor(red) bin(15))  || ///
				   (histogram PP_1 if dltv==0, start(0) fcolor(none) lcolor(black)  bin(15)) ///
				   if `touse', ///
				   title("Estimated Probability of Treatment (`Country')") legend(order(1 "Treated (dltv=20)" 2 "Control (dltv==0)"))
				   graph export "../Charts/Pr_Treat_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'(dltv=20)_v`str_data_version'.emf", as(emf) replace
		}
	}
	
	// Statistics of the propensity score (PP)
	tabstat PP   if `touse', by(dltv) stat(min p1 p50 p99 max mean sd sum n)
	tabstat PP_w if `touse', by(dltv) stat(min p1 p50 p99 max mean sd sum n)
	*di "Spec = `Spec'"
	*exit
	
	*-------------------------------------------------------------------------------
	* Inverse Propensity-Score Weights
	*-------------------------------------------------------------------------------
	gen IPSw   = 1/PP
	gen IPSw_w = 1/PP_w 
	label var IPSw   "1/PP: Inverse propensity-score weights " 
	label var IPSw_w "1/PP_w: winsorized inverse propensity-score weights " 
	su IPSw  IPSw_w

	save IPW_Data.dta, replace
}

*===============================================================================
* Step 3: AIPW estimation (AIPW`cnt') & FE estimation
* AIPW-steps: 
* 1) Obtain aipwt`cnt' = ipwt`cnt' + mdifft`cnt') for each obs,
*     where mdifft`cnt' involves estimating the outcome model (mu`cnt') 
* 2) Take the average of aipwt`cnt' to get AIPW`cnt' and its standard errors,
*     following Jorda and Taylor (2015)
* FE-step. Estimation.
*===============================================================================
// Load data with TM results, if Steps 1-2 are skipped
if `TM_estimation' == 0 {
	use IPW_Data, clear
}

//[Outcome variable] Note: d4 and _w1 will be automatically added below, e.g., d4lnrhhdebt_w1 d4lnrpvcon_w1
local y_list lnrhhdebt lnrpvcon lnrgdp lnrp_house lncpi ///
	x_lnrbankmort x_lnrhhdebtexbm ///
	x_lnrbankloan x_lnrnonbankloan x_lnrbanknonmort ///
	w_lnrmortdebt w_lnrnonmortdebt ///
	x_lnrbankloan_b x_lnrbankmort_b x_lnrbanknonmort_b ///
	x_share_b_hhdebt x_share_bm_bloan x_share_m_hhdebt 
*local y_list lnrhhdebt w_lnrmortdebt w_lnrnonmortdebt


local max_h 8                  //[Horizons]
if "`bin'" == "3bin" {
	local dltv_list -10 10     //[Treatments] 
	local dltv_label "T10 E10" //[Treatment labels - T: tightening, E: easing]
}	
if "`bin'" == "4bin" | "`bin'" == "5bin" {	
	local dltv_list -20 -10 10     //[Treatments] Note that we will not estimate ATE for dltv == 20 because of the small number of obs (although the treatment model is fitted for all).
	local dltv_label "T20 T10 E10" //[Treatment labels - T: tightening, E: easing]
}

*----------------------------------------
* Temporary dta file to save results
*---------------------------------------- 
preserve
	local FileName AIPW_results.dta
	clear 
	set obs 1
	gen h = .
	save `FileName', replace
restore

*----------------------------------------
* Matrixes to store estimated AIPWs
*---------------------------------------- 
// Prepare a matrix "m_APIW" to store results
local max_h_plus1 = `max_h' + 1
local n_treatment: word count `dltv_list' 
matrix define m_AIPW = J(`max_h_plus1',`n_treatment',.)
// Column names of m_AIPW
matrix colnames m_AIPW = `dltv_label'
// Row names of m_AIPW
local m_AIPW_rownames ""
forvalues h = 1/`max_h_plus1' {
	local m_AIPW_rownames `m_AIPW_rownames' h`h'
}
matrix rownames m_AIPW = `m_AIPW_rownames'
// Check m_AIPW
mat list m_AIPW

*-------------------------------------------------------------------------------
* WLS to obtain the predicted outcome for each treatment group
* (mu0, mu1, mu2, mu3) for each horizon (h)
*-------------------------------------------------------------------------------
// Output excel file for FE
	local FE_myfile="../Results/Macropru/LTV/FE_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'`FElagon'_v`str_data_version'.xls"
    local FE_app_or_rep replace	
	
// Loop over y variables
*local y_list lnrpvcon
foreach y of local y_list {
	di "======================================================================="
	di " AIPW on `y' for horizons 1 - `max_h'  "
	di "======================================================================="	
	di "=================== AIPW_`y': Refresh matrixes to save results ================="
	matrix define AIPW_b        = m_AIPW
	matrix define AIPW_se       = m_AIPW
	matrix define AIPW_p        = m_AIPW
	matrix define AIPW_n_b      = m_AIPW
	matrix define AIPW_n_se     = m_AIPW
	matrix define AIPW_n_p      = m_AIPW
	matrix define AIPW_n_dltva  = m_AIPW
	matrix define AIPW_obs_bin  = m_AIPW
	matrix define AIPW_obs2_bin = m_AIPW
	mat list AIPW_b  // to check
	mat list AIPW_n_b  // to check
	
	di "=================== AIPW_`y': Loop for horizons ================="
	*local max_h = 1
	forvalues h = 1/`max_h' {
		// Predicted outcome variables
		capture drop mu*
		gen mu0=.
		gen mu1=.
		gen mu2=.
		if "`bin'" == "3bin" {
			di as err "Outcome regressions: `bin'" 	
			label var mu0 "Predicted outcome of non-treated (dltv = 0)"
			label var mu1 "Predicted outcome of treated (dltv = -10)"
			label var mu2 "Predicted outcome of treated (dltv = 10)"
		}	
		if "`bin'" == "4bin" | "`bin'" == "5bin" {
			di as err "Outcome regressions: `bin'" 	
			gen mu3=.
			label var mu0 "Predicted outcome of non-treated (dltv = 0)"
			label var mu1 "Predicted outcome of treated (dltv = -20)"
			label var mu2 "Predicted outcome of treated (dltv = -10)"
			label var mu3 "Predicted outcome of treated (dltv = 10)"
		}
		
		di "---------------- [h=`h'] Outcome variable: D`h'_`y' ----------------"
		// Outcome variable: cumulative growth for the horizon h
		gen D`h'_`y' = (F`h'.`y' - `y')*100
		label var D`h'_`y' "100*(y_t+`h' - y_t): (y =`y')not winsorized"
		// Outcome variable: winsorized cumulative growth for the horizon h
		winsor D`h'_`y', gen(D`h'_`y'_w) p(0.01)
		label var D`h'_`y'_w "100*(y_t+`h' - y_t): (y =`y') winsorized"

		di "---------------- [h=`h'] Predicted outcome: Control (mu0) ----------------"
		// Control: mu0 (dltv == 0)  l.d4lnrhhdebt_w1
		*reg D`h'_`y'_w   l(0/1).d1`y'_w1 l(0/1).strate l(0/1).d1lnrgdp_w1 l(0/1).vix i.qr AE EM [pweight = IPSw_w] ///
		*reg D`h'_`y'_w   l(0/1).d4`y'_w1 l(0/1).strate l(0/1).d4lnrgdp_w1 l(0/1).vix i.qr AE EM [pweight = IPSw_w] ///
		di "cmd:reg D`h'_`y'_w   l(0/0).d4`y'_w1 l(0/0).strate l(0/0).d4lnrgdp_w1 l(0/0).vix `controls_OM' i.qr AE [pweight = IPSw_w] if `touse' & dltv == 0,  cluster(isocode)"
		reg D`h'_`y'_w   l(0/0).d4`y'_w1 l(0/0).strate l(0/0).d4lnrgdp_w1 l(0/0).vix `controls_OM' i.qr AE [pweight = IPSw_w] ///
			if `touse' & dltv == 0,  cluster(isocode)
		capture drop temp
		predict temp
		replace mu0 = temp if `touse' 

		// Treated: for 4 or 5 bins, we have: mu1 mu2 mu3 (dltv != 0)	
		// and: local dltv_list -20 -10 10
		local cnt 0
		foreach dltv_value of local dltv_list { 
			local cnt = `cnt' + 1
			di "---------------- [h=`h'] Predicted outcome: Treated (mu`cnt') ----------------"
			*local cnt = 1
			*---------------------------------------------------------------------------
			* IPW term of the AIPW estimator (ipwt`cnt') for each t
			*---------------------------------------------------------------------------
			di "cmd:gen ipwt`cnt'   = D`h'_`y'_w*(flg_dltv`cnt'/p`cnt'_w - flg_dltv0/p0_w) if `touse'"
			gen ipwt`cnt'   = D`h'_`y'_w*(flg_dltv`cnt'/p`cnt'_w - flg_dltv0/p0_w) if `touse'
			gen ipwt`cnt'_n = ipwt`cnt'/(`dltv`cnt'_a_local')
			label var ipwt`cnt'   "IPW term for the AIPW estimator of (dltv == `dltv_value')" 
			label var ipwt`cnt'_n "ipwt`cnt'/dltv`cnt'_a (dltv == `dltv_value')" 
			
			*---------------------------------------------------------------------------
			* Adjustment term of the AIPW estimator (mdifft`cnt') for each t
			*---------------------------------------------------------------------------	
			// Outcome model regression (WLS estimation)
			*di "cmd: reg D`h'_`y'_w   d1`y'_w1 strate d1lnrgdp_w1 vix i.qr i.isocode [pweight = IPSw_w] ///" 
			* reg D`h'_`y'_w   l(0/1).d1`y'_w1 l(0/1).strate l(0/1).d1lnrgdp_w1 l(0/1).vix i.qr AE EM [pweight = IPSw_w] ///
			* reg D`h'_`y'_w   l(0/1).d4`y'_w1 l(0/1).strate l(0/1).d4lnrgdp_w1 l(0/1).vix i.qr AE EM [pweight = IPSw_w] ///
			di "cmd:reg D`h'_`y'_w   l(0/0).d4`y'_w1 l(0/0).strate l(0/0).d4lnrgdp_w1 l(0/0).vix `controls_OM' i.qr AE [pweight = IPSw_w] if `touse' & dltv == `dltv_value',  cluster(isocode)"
			capture reg D`h'_`y'_w   l(0/0).d4`y'_w1 l(0/0).strate l(0/0).d4lnrgdp_w1 l(0/0).vix `controls_OM' i.qr AE [pweight = IPSw_w] ///
				if `touse' & dltv == `dltv_value',  cluster(isocode)
			if _rc continue
			local dltv`cnt'_obs_local = e(N)
			capture drop temp
			predict temp
			replace mu`cnt' = temp if `touse' 
			
			// Adjustment term for the AIPW (c.f. Stata document for teffects)
			gen mdifft`cnt'   = - mu`cnt'*((flg_dltv`cnt' - p`cnt'_w)/p`cnt'_w) ///
							    + mu0*((flg_dltv0 - p0_w)/p0_w) if `touse'
			gen mdifft`cnt'_n = mdifft`cnt'/(`dltv`cnt'_a_local')
			label var mdifft`cnt'   "Adjustment term for the AIPW estimator of (dltv == `dltv_value')" 
			label var mdifft`cnt'_n "mdifft`cnt'/dltv`cnt'_a (dltv == `dltv_value')" 
			
			*---------------------------------------------------------------------------
			* AIPW_t (aipwt`cnt' = ipwt`cnt' + mdifft`cnt') for each t
			*---------------------------------------------------------------------------	
			gen aipwt`cnt'   = ipwt`cnt'   + mdifft`cnt'
			gen aipwt`cnt'_n = ipwt`cnt'_n + mdifft`cnt'_n
			label var aipwt`cnt'   "Whole term for the AIPW estimator of (dltv == `dltv_value')" 
			label var aipwt`cnt'_n "aipwt`cnt'/dltv`cnt'_a (dltv == `dltv_value')" 		
			
			*---------------------------------------------------------------------------
			* AIPW estimates (AIPW`cnt') w/o size control LTV changes
			*---------------------------------------------------------------------------	
			di "dLTV`cnt'"
			// [FYI] AIPW 
			gen AIPW = 1 // constant for convenience in next reg to get mean
			reg aipwt`cnt' AIPW if `touse', nocons cluster(isocode)
			mat rtable = r(table) // p-value				
			eststo AIPW`cnt'_h`h'			
			
			// Save results
			mat AIPW_b[`h',`cnt']  = _b[AIPW]
			mat AIPW_se[`h',`cnt'] = _se[AIPW]	
			mat AIPW_p[`h',`cnt']  = rtable[4,1]	
			drop AIPW
			
			*---------------------------------------------------------------------------
			* AIPW estimates (AIPW`cnt') per 1 ppt change
			*---------------------------------------------------------------------------			
			// [Baseline] AIPW per 1 ppt change in the LTV limit
			gen AIPW = 1 // constant for convenience in next reg to get mean
			reg aipwt`cnt'_n AIPW if `touse', nocons cluster(isocode)
			mat rtable = r(table) // p-value				
			eststo AIPW`cnt'_n_h`h'
			
			// [Baseline] AIPW per 1 ppt change in the LTV limit
			mat AIPW_n_b[`h',`cnt']      = _b[AIPW]
			mat AIPW_n_se[`h',`cnt']     = _se[AIPW]	
			mat AIPW_n_p[`h',`cnt']      = rtable[4,1]
			mat AIPW_n_dltva[`h',`cnt']  = `dltv`cnt'_a_local'
			mat AIPW_obs_bin[`h',`cnt']  = `dltv`cnt'_obs_local'
			mat AIPW_obs2_bin[`h',`cnt'] = `dltv`cnt'_obs2_local'
			drop AIPW
			
		} // for treatment level (cnt)
		capture drop aipwt* 
		capture drop ipwt* 
		capture drop mdifft* 
		capture drop mu*
	} // for horizon h


	*-------------------------------------------------------------------------------
	* Fixed effect regression
	*-------------------------------------------------------------------------------
	// var1: dependent variable
	local var1 d4`y'_w1
	
	// Specify the list of country groups
	// Run subsample estimation ONLY for hhdebt and pvcon.
	local condlist ALL
		if "`var1'"=="d4lnrhhdebt_w1" {	
				local condlist ALL AE EM
		}
		if "`var1'"=="d4lnrpvcon_w1" {	
				local condlist ALL AE EM
		}
	
	foreach var3 in `condlist' {
		di "==============================================="
		di " Non-linear effects of LTV_average "
		di " ----------------------------------------------"
		di " Yvar: `var1'"
		di " ----------------------------------------------"
		di " Sample countries: `var3' "
		di "==============================================="	
	
		// Macro controls
		if "`var1'"=="d4lnrgdp_w1" {
		    if `flg_FE_lags_on' == 0 local OtherReg strate `controls_FE'
		    if `flg_FE_lags_on' == 1 local OtherReg l.strate `controls_FE' 
		}
		else {
			if `flg_FE_lags_on' == 0 local OtherReg strate d4lnrgdp_w1 `controls_FE'
			if `flg_FE_lags_on' == 1 local OtherReg l.strate l.d4lnrgdp_w1 `controls_FE'
		}
												
		// Regression
		if `flg_FE_lags_on' == 1 {
		    
			// List of regressors
			if "`bin'" == "5bin" {	
				local regressors l.`var1' `OtherReg' l(1/4).(dltv_level_T10 dltv_level_T20 dltv_level_L10 dltv_level_L20)
			}
			if "`bin'" == "4bin" {
				local regressors l.`var1' `OtherReg' l(1/4).(dltv_level_T10 dltv_level_T20 dltv_level_L10)
			}
			if "`bin'" == "3bin" {
				local regressors l.`var1' `OtherReg' l(1/4).(dltv_level_T10 dltv_level_L10)
			}
			
			di "cmd: areg `var1' `regressors' i.dateq if `touse' & `var3'==1, absorb(ifscode) clu(ifscode)"
			areg `var1' `regressors' i.dateq if `touse' & `var3'==1, absorb(ifscode) clu(ifscode)
			
			local n_countries = e(N_clust)
			
			// Number of observations in bins
			gen flg_insample = e(sample)
			local cnt 0
			local l1 = .
			local l2 = . 
			foreach dltv_value of local dltv_list { 
				local cnt = `cnt' + 1				
				count if dltv == `dltv_value' & (F1.flg_insample == 1 | F2.flg_insample == 1 | F3.flg_insample == 1 | F4.flg_insample == 1)
				local dltv`cnt'_obs_local = r(N)
				if `dltv_value' == -10 local l1 =  `dltv`cnt'_obs_local'
				if `dltv_value' == -20 local l2 =  `dltv`cnt'_obs_local'
			}
			drop flg_insample
			
			local dltv_v dltv_level_T10
			local m1=_b[l.`dltv_v']+_b[l2.`dltv_v']+_b[l3.`dltv_v']+_b[l4.`dltv_v']
			test l.`dltv_v' + l2.`dltv_v' + l3.`dltv_v' +l4.`dltv_v' = 0
			local n1=r(p)
			
			if "`bin'" == "4bin" | "`bin'" == "5bin" {	
				local dltv_v dltv_level_T20
				local m2=_b[l.`dltv_v']+_b[l2.`dltv_v']+_b[l3.`dltv_v']+_b[l4.`dltv_v']
				test l.`dltv_v' + l2.`dltv_v' + l3.`dltv_v' +l4.`dltv_v' = 0
				local n2=r(p)
			}
			else {
				local m2 = .
				local n2 = . 
			}
			
			local dltv_v dltv_level_L10
			local m3=_b[l.`dltv_v']+_b[l2.`dltv_v']+_b[l3.`dltv_v']+_b[l4.`dltv_v']
			test l.`dltv_v' + l2.`dltv_v' + l3.`dltv_v' +l4.`dltv_v' = 0
			local n3=r(p)
						
		}
		if `flg_FE_lags_on' == 0 {
		    			
			// List of regressors
			if "`bin'" == "5bin" {	
				local regressors `var1' `OtherReg' dltv_level_T10 dltv_level_T20 dltv_level_L10 dltv_level_L20  
			}
			if "`bin'" == "4bin" {
				local regressors `var1' `OtherReg' dltv_level_T10 dltv_level_T20 dltv_level_L10
			}
			if "`bin'" == "3bin" {
				local regressors `var1' `OtherReg' dltv_level_T10 dltv_level_L10
			}
			
			di "cmd: areg D4_`y'_w `regressors' i.dateq if `touse' & `var3'==1, absorb(ifscode) clu(ifscode)"
			areg D4_`y'_w `regressors' i.dateq if `touse' & `var3'==1, absorb(ifscode) clu(ifscode)
			
			local n_countries = e(N_clust)
			
			// Number of observations in bins
			local cnt 0
			local l1 = .
			local l2 = . 
			foreach dltv_value of local dltv_list { 
				local cnt = `cnt' + 1
				count if e(sample) & dltv == `dltv_value'
				local dltv`cnt'_obs_local = r(N)
				if `dltv_value' == -10 local l1 =  `dltv`cnt'_obs_local'
				if `dltv_value' == -20 local l2 =  `dltv`cnt'_obs_local'
			}
			
			local dltv_v dltv_level_T10
			local m1=_b[`dltv_v']
			test `dltv_v' = 0
			local n1=r(p)
			
			if "`bin'" == "4bin" | "`bin'" == "5bin" {	
				local dltv_v dltv_level_T20
				local m2=_b[`dltv_v']
				test `dltv_v' = 0
				local n2=r(p)
			}
			else {
				local m2 = .
				local n2 = . 
			}
	
			local dltv_v dltv_level_L10
			local m3=_b[`dltv_v']
			test `dltv_v' = 0
			local n3=r(p)			
		}
		
		outreg2  using `FE_myfile', excel `FE_app_or_rep' drop(i.dateq) `addStatAREG' addtext(Errors, robust, TIME FE, YES, COU FE, YES, Period, >=`StartYear', Group, `var3',Country no, `n_countries', Sum coeff LTV -10, `m1', P-value -10, `n1', Obs LTV -10, `l1', Sum coeff LTV -20, `m2', P-value -20, `n2', Obs LTV -20, `l2')
		
		if "`FE_app_or_rep'" == "replace" local FE_app_or_rep append
			
		// Save results in AIPW_* matrices when estimating for "ALL"
		if "`var3'" == "ALL" {
			if "`bin'" == "3bin" {	
				// tightening
				mat AIPW_n_b[`max_h_plus1',1] = `m1'
				mat AIPW_n_p[`max_h_plus1',1] = `n1'			
				// loosening
				mat AIPW_n_b[`max_h_plus1',2] = `m3'
				mat AIPW_n_p[`max_h_plus1',2] = `n3'
			}
			if "`bin'" == "4bin" | "`bin'" == "5bin" {	
				// large tightening (in order of the loop above)
				mat AIPW_n_b[`max_h_plus1',1] = `m2'
				mat AIPW_n_p[`max_h_plus1',1] = `n2'
				// small tightening
				mat AIPW_n_b[`max_h_plus1',2] = `m1'
				mat AIPW_n_p[`max_h_plus1',2] = `n1'			
				// loosening
				mat AIPW_n_b[`max_h_plus1',3] = `m3'
				mat AIPW_n_p[`max_h_plus1',3] = `n3'
			}
			
			local cnt 0
			foreach dltv_value of local dltv_list { 
				local cnt = `cnt' + 1
				mat AIPW_obs_bin[`max_h_plus1',`cnt']  = `dltv`cnt'_obs_local'
			}
		}

	} // End of the loop for var3 (ALL, AE, or EM)
		
	*-------------------------------------------------------------------------------
	* Results
	*-------------------------------------------------------------------------------
	di "=================== AIPW_`y': [FYI] Results w/o size control ================="
	esttab AIPW1_h* 
	esttab AIPW2_h* 
	if "`bin'" == "4bin" | "`bin'" == "5bin" esttab AIPW3_h*
	mat list AIPW_b
	mat list AIPW_p
	mat list AIPW_se
	di "=================== AIPW_`y': Baseline results (per 1 ppt change) ================="
	esttab AIPW1_n_h* 
	esttab AIPW2_n_h* 
	if "`bin'" == "4bin" | "`bin'" == "5bin" esttab AIPW3_n_h*
	mat list AIPW_n_b
	mat list AIPW_n_p
	mat list AIPW_n_se	
	*}
	*br isocode dateq ipwt* mdifft* aipwt* dltv if dltv == 0
	*exit
	*-------------------------------------------------------------------------------
	* Saving results (`FileName'.dta)
	*-------------------------------------------------------------------------------
	di "=================== AIPW_`y': Outsheet results ================="
	preserve
	clear
		// Convert matrix to variables (constructed as a "wide" panel data set)
		svmat AIPW_b,        names(matcol)
		svmat AIPW_se,       names(matcol)
		svmat AIPW_p,        names(matcol)		
		svmat AIPW_n_b,      names(matcol)
		svmat AIPW_n_se,     names(matcol)
		svmat AIPW_n_p,      names(matcol)
		svmat AIPW_n_dltva,  names(matcol)
		svmat AIPW_obs_bin,  names(matcol)
		svmat AIPW_obs2_bin, names(matcol)

		gen Outcome     = "`y'"  // F4_d4lnrhhdebt_w1 etc
		gen h = _n               // Horizion
		replace h = 99 if h == `max_h_plus1'
		
		// Save
		append using `FileName'
		drop if h == .
		sort Outcome h
		save `FileName', replace	
	restore
} // for y-variable

*===============================================================================
* Outsheet results (`FileName'.dta)
*===============================================================================
preserve
	use `FileName', clear
	export excel Outcome h AIPW_n_b* AIPW_n_p* AIPW_b* AIPW_p* AIPW_n_dltva* AIPW_obs_bin* AIPW_obs2_bin* ///
	using "../Results/Macropru/AIPW_`bin'_`Spec'_`today'`Controls'`NonIntro'`NonCRE'`NonNarrow'`FElagon'_v`str_data_version'.xlsx", replace firstrow(variables)
restore

di "========== dltv_var is `dltv_var' =========="
di "cmd: version 16: table dltv if `touse', c(count `dltv_var' mean `dltv_var' sd `dltv_var')"
version 16: table dltv if `touse', c(count `dltv_var' mean `dltv_var' sd `dltv_var')
version 16: table dltv if AE == 1 & `touse' , c(count `dltv_var' mean `dltv_var' sd `dltv_var')
version 16: table dltv if EM == 1 & `touse' , c(count `dltv_var' mean `dltv_var' sd `dltv_var')


// Erase temporary dta files
//erase `FileName'
//erase IPW_Data.dta
**********************************
log close
exit
