!	----------------------------------------------------------------------
!	Module name: SimulationModule.f90
!
!	collection of sub-programs to simulate artificial data
!	----------------------------------------------------------------------



Module Transitions


	use Globals
	use LinInterpModule
	use OptimizationMod
    use sorting
    use sorting2
   
	
	implicit none


	integer:: xseed
	type Individual
		real(8) :: A, AS, HR, WORK, WLTH, CON, INC, LINC, KINC, EL, VF
		integer :: X
	end type Individual
	type(Individual), dimension(:), allocatable :: Worker
	real, external:: sunif, snorm
	!	sunif is okay with double precision but snorm should be single precision

	
	real(8), dimension(na,nx,nk,4):: cscoef_vf_k
    real(8), dimension(na,nx):: VF_time
	real(8):: vf_work, vf_home, as_work, as_home, constraint, xmin, cscoef_vf_a(na,4), erhs, lbd ,ubd, ct, hr_home, hr_work, ap_hour
	real(8), external:: csval


contains



!	----------------------------------------------------------------------
!	File name: SimulateData.f90
!
!	generate artificial data for aggregate capital, interest rate, and 
!	wage rate through simulating individuals' decision rules. These data
!	will be used in RegressLOM().
!	----------------------------------------------------------------------


subroutine Compute_TR()


	


	!	set the seed number for random number generator

	xseed = 2


	!	allocate space for cross section data generation

	allocate(Worker(Nindiv))


	!	intialize distribution of workers

	do indiv = 1, Nindiv
		Worker(indiv)%A = kss
		Worker(indiv)%X = mod(indiv, nx) + 1
	end do


	!	initial aggregate capital

	Kdata(1) = sum(Worker%A)/Nindiv
	

	


	!	start generating artificial time series data

	do time = 1, Nperiod


		!	interpolate value function along K-dimension

		do ix = 1, nx

			do ia = 1, na

				call csint(nk, kgrid, PF(ia,ix,:,GIdata(time)), cscoef_vf_k(ia,ix,:,:))

			end do

        end do

        !	evaluate VFE_time_p at Kdata(t+1)
        
       Kdata(time+1) = min(max(dexp(Kcoef(1) + Kcoef(2)*dlog(Kdata(time)) + Kcoef(3)*dlog(Gdata(time))), Kgrid(1)), Kgrid(nk))
				

		do ix = 1, nx            
			do ia = 1, na

				VF_time(ia,ix) = csval(nk, Kdata(time+1), kgrid, cscoef_vf_k(ia,ix,:,:))

            end do
        end do


		 
         
        ffr   = dexp(Rcoef(1) + Rcoef(2)*dlog(Kdata(time)) + Rcoef(3)*dlog(Gdata(time)))          
        price   = dexp(Pcoef(1) + Pcoef(2)*dlog(Kdata(time)) + Pcoef(3)*dlog(Gdata(time)))  
        mc   = dexp(Mcoef(1) + Mcoef(2)*dlog(Kdata(time)) + Mcoef(3)*dlog(Gdata(time)))  
        wage   = dexp(Wcoef(1) + Wcoef(2)*dlog(Kdata(time)) + Wcoef(3)*dlog(Gdata(time)))  
        r_k = mc**(1.0D0/alpha)*alpha*(wage/(1.0D0-alpha))**((alpha-1.0D0)/alpha) - delta 
        
        

		!	simulate cross-section data

		do indiv = 1, Nindiv

			call csint(na, agrid, VF_time(:,Worker(indiv)%X), cscoef_vf_a)
            
            
                ! Working
                work = 1.0D0			    
                constraint = (1.0D0+r_k)*Worker(indiv)%A + divd
                    
                lbd = amin+divd
				ubd = min(constraint + wage*gh(1.0D0)*exgrid(Worker(indiv)%X), amax)
				call fmin(NegValueF, lbd, ubd, xmin)			    
                    
				vf_work = ValueF(xmin)
                as_work = xmin
                hr_work = hour
                
                
                !	not-working
			
				work = 0.0D0
				constraint = min((1.0D0+r_k)*Worker(indiv)%A + divd, amax)
                    
                lbd = amin+divd
				ubd = constraint
				call fmin(NegValueF, lbd, ubd, xmin)			    
				               				    
				vf_home = ValueF(xmin)
                as_home = xmin
                hr_home = 0.0D0
                
                if (vf_work >= vf_home) then

				    Worker(indiv)%VF = vf_work
                    Worker(indiv)%AS = as_work
                    Worker(indiv)%HR = hr_work                   
                    Worker(indiv)%EL = exgrid(Worker(indiv)%X)*gh(Worker(indiv)%HR)
                    Worker(indiv)%WORK = 1.0D0
                    
                    
                    else                                           
                    
                    Worker(indiv)%VF = vf_home
                    Worker(indiv)%AS = as_home
                    Worker(indiv)%HR = 0.0D0
                    Worker(indiv)%EL = 0.0D0
                    Worker(indiv)%WORK = 0.0D0
                                       

                    end if
                    
                    Worker(indiv)%WLTH = Worker(indiv)%A
                    Worker(indiv)%INC = r_k*Worker(indiv)%A + wage*gh(Worker(indiv)%HR)*exgrid(Worker(indiv)%X)
                    Worker(indiv)%LINC = wage*gh(Worker(indiv)%HR)*exgrid(Worker(indiv)%X)
                    Worker(indiv)%KINC = r_k*Worker(indiv)%A
                    Worker(indiv)%CON = Worker(indiv)%INC + Worker(indiv)%A  + divd - Worker(indiv)%AS

		end do


		!	generate time series data for aggregate macro variables

		!Kdata(time+1) = min(max(sum(Worker%AS)/Nindiv, kgrid(1)), kgrid(nk))
        Kdata(time+1) = sum(Worker%AS)/Nindiv
		Wdata(time) = wage
		Rdata(time) = irate 
		Edata(time) = sum(Worker%WORK)/Nindiv
        Ldata(time) = sum(Worker%EL)/Nindiv
        Hdata(time) = sum(Worker%HR)/Nindiv
		Cdata(time) = sum(Worker%CON)/Nindiv
		Ydata(time) = (Kdata(time)**alpha)*(Ldata(time)**(1.0D0-alpha)) - fcost
		Idata(time) = Kdata(time+1) - (1.0D0-delta)*Kdata(time)
        Rfdata(time) = ffr   
        Mdata(time) = mc
        Pdata(time) = price
		
		! Compute TR
        
         call CalculateINEQ()   

		!	prepare for the simulation in the next period

		Worker%A = Worker%AS
		call NextIdiosyncraticShock()


		!	show how simulation is going on

		if (mod(time, 20) == 0) then
			print *, "time = ", time	
			print *, "G    = ", Gdata(time)            
			print *, "K    = ", Kdata(time)
			print *, "Kp   = ", Kdata(time+1)
			print *, "E    = ", Edata(time)
            print *, "H    = ", Hdata(time)
			print *, "L    = ", Ldata(time)
			print *, "Y    = ", Ydata(time)
			print *, "C    = ", Cdata(time)
			print *, "W    = ", Wdata(time)
            print *, "P    = ", Pdata(time)
			print *, "D    = ", Ddata(time)       
            print *, "FFR  = ", Rfdata(time)       
            print *, "MC   = ", Mdata(time)   			
            print *, " "
        end if


	end do


	!	deallocate memory

	deallocate(Worker)
	

end subroutine





subroutine NextIdiosyncraticShock()


	real(8):: ushock


	!	generate next period individual productivity shock

	do indiv = 1, Nindiv

		ushock = sunif(xseed)

		do jx = 1, nx

			if (ushock <= CtrX(Worker(indiv)%X,jx)) then
				Worker(indiv)%X = jx
				exit
			end if
		
		end do

	end do


end subroutine





real(8) function SolveHour(xhour)

	implicit none

	real(8), parameter:: tiny = 1.0D-20, huge = 1.0D30
	real(8) cons, xhour

	
    cons = constraint + wage*gh(xhour)*exgrid(Worker(indiv)%X) - ap_hour
    
    if (cons <= 0.0D0) then
        SolveHour=  -huge 
    else
	    SolveHour =  dlog(cons) - B*xhour**(1.0D0+1.0D0/gama)/(1.0D0+1.0D0/gama)
    end if

end function



real(8) function ValueFh(xhour)

	implicit none

	real(8) xhour

	ValueFh = SolveHour(xhour) + bta*csval(na, ap_hour, agrid, cscoef_vf_a)

end function




real(8) function NegValueFh(xhour)

	implicit none

	real(8) xhour

	NegValueFh = -ValueFh(xhour)

end function


real(8) function Utility(ap)

	implicit none

	real(8), parameter:: tiny = 1.0D-20, tol_hour = 1.0D-5, huge = 1.0D30
	real(8) ap, xhour_min, cm
	
	if (work == 1.0D0) then
	ap_hour = ap

	call fmin(NegValueFh, hbar, 0.7D0, xhour_min)   

    hour = xhour_min
    cm = constraint + wage*gh(hour)*exgrid(Worker(indiv)%X)  - ap
    
    else
        
    hour = 0.0D0
    cm = constraint  - ap
    
    end if
    

    if (cm <= 0.0D0) then 
        Utility = -huge
    else
	    Utility = dlog(cm) - B*hour**(1.0D0+1.0D0/gama)/(1.0D0+1.0D0/gama)
    end if



end function



real(8) function ValueF(ap)

	implicit none

	real(8) ap

	ValueF = Utility(ap) + bta*csval(na, ap, agrid, cscoef_vf_a)

end function


real(8) function NegValueF(ap)

	implicit none

	real(8) ap

	NegValueF = -ValueF(ap)

end function




real(8) function gh(h)

	implicit none

	real(8) h

	gh = max(h-hbar, 0.0D0)

end function




        
  subroutine CalculateINEQ()   
  
  
  real(8), dimension(Nindiv,4) :: asst, incm, kinc, linc, hours , con
  real(8), dimension(Nindiv) :: asst_y, inc_y, kinc_y, linc_y, hours_y, con_y, linc_w_y, tem_w, cdf_y, pdf, phours_y, qt_y, pqt_y
  real(8) :: temp_e(4,Nindiv), temp_h(25,Nindiv)
     
  
  
  
            if (mod(time,4) == 1) then

                do indiv = 1, Nindiv   

                asst(indiv,1)=Worker(indiv)%WLTH
                incm(indiv,1)=Worker(indiv)%INC
                linc(indiv,1)=Worker(indiv)%LINC
                kinc(indiv,1)=Worker(indiv)%KINC
                hours(indiv,1)=Worker(indiv)%HR
                con(indiv,1)=Worker(indiv)%CON
                
                end do

            else if (mod(time,4) == 2) then
                
                do indiv = 1, Nindiv  

                asst(indiv,2)=Worker(indiv)%WLTH
                incm(indiv,2)=Worker(indiv)%INC
                linc(indiv,2)=Worker(indiv)%LINC
                kinc(indiv,2)=Worker(indiv)%KINC
                hours(indiv,2)=Worker(indiv)%HR
                con(indiv,2)=Worker(indiv)%CON
                
                end do
                
            else if (mod(time,4) == 3) then
                
                do indiv = 1, Nindiv                  
                
                asst(indiv,3)=Worker(indiv)%WLTH
                incm(indiv,3)=Worker(indiv)%INC
                linc(indiv,3)=Worker(indiv)%LINC
                kinc(indiv,3)=Worker(indiv)%INC
                hours(indiv,3)=Worker(indiv)%HR
                con(indiv,3)=Worker(indiv)%CON
                
                end do

            else if (mod(time,4) == 0) then
                
                time_y=time/4
                                                
                do indiv = 1, Nindiv                
          
                asst(indiv,4)=Worker(indiv)%WLTH
                incm(indiv,4)=Worker(indiv)%INC
                linc(indiv,4)=Worker(indiv)%LINC
                kinc(indiv,4)=Worker(indiv)%KINC
                hours(indiv,4)=Worker(indiv)%HR
                con(indiv,4)=Worker(indiv)%CON
                
                
                ! Make annual Data
                       
                asst_y(indiv)=asst(indiv,4) ! stock variable
                inc_y(indiv)=sum(incm(indiv,:))
                linc_y(indiv)=sum(linc(indiv,:))
                kinc_y(indiv)=sum(kinc(indiv,:))
                hours_y(indiv)=sum(hours(indiv,:))              
                con_y(indiv)=sum(con(indiv,:))              
                                                                                                                                              
                end do 
                
               
                
                
                tem_w = asst_y
                linc_w_y = linc_y
                
                
               
               call dsorting2(Nindiv,tem_w,linc_w_y)                                          
               call dsort(asst_y,Nindiv)               
               call dsort(linc_y,Nindiv)
               call dsort(kinc_y,Nindiv)
               call dsort(inc_y,Nindiv)
               call dsort(hours_y,Nindiv)
               call dsort(con_y,Nindiv)
               
                
               
               
           ! Gini Coefficient 
           
            
            pdf = 1.0D0/Nindiv                    
            
                        
            call dist1(asst_y, pdf, 5, TWLTH(:,time_y), GWdata(time_y))  
            call dist1(con_y, pdf, 5, TCON(:,time_y), GCdata(time_y))  
            call dist1(inc_y, pdf, 5, TINC(:,time_y), GYdata(time_y))  
            call dist1(linc_y, pdf, 5, TLINC(:,time_y), GEdata(time_y))  
            call dist1(linc_w_y, pdf, 5, TLINC_W(:,time_y), GEWdata(time_y))  
            call dist1(hours_y, pdf, 5, THOURS(:,time_y), GHdata(time_y))  
            call dist0(hours_y, pdf, 5, THOURS_Z(:,time_y), GHZdata(time_y))  
            
            
            ! Employment Transition
            
            temp_e = 0.0D0
            
            do indiv = 1, Nindiv  
            if (phours_y(indiv) > 0.0D0 .and. hours_y(indiv) > 0.0D0) then
                temp_e(1,indiv) = 1.000 
            elseif (phours_y(indiv) > 0.0D0 .and. hours_y(indiv) == 0.0D0) then
                temp_e(2,indiv) = 1.000 
            elseif (phours_y(indiv) == 0.0D0 .and. hours_y(indiv) > 0.0D0) then
                temp_e(3,indiv) = 1.000 
            elseif (phours_y(indiv) == 0.0D0 .and. hours_y(indiv) == 0.0D0) then
                temp_e(4,indiv) = 1.000 
            end if            
            end do
            
            TR_E(:,time_y) = sum(temp_e,2)/Nindiv             
            phours_y = hours_y      
            
            
            
            ! Hours Transition
            
            do indiv = 1, Nindiv             
            if (hours_y(indiv) <=hours_y(0.2*Nindiv)) then
                qt_y(indiv) = 1.0D0
                else if (hours_y(0.2*Nindiv) < hours_y(indiv) .and. hours_y(indiv) <= hours_y(0.4*Nindiv)) then
                qt_y(indiv) = 2.0D0
                else if (hours_y(0.4*Nindiv) < hours_y(indiv) .and. hours_y(indiv) <= hours_y(0.6*Nindiv)) then
                qt_y(indiv) = 3.0D0
                else if (hours_y(0.6*Nindiv) < hours_y(indiv) .and. hours_y(indiv) <= hours_y(0.8*Nindiv)) then
                qt_y(indiv) = 4.0D0
                else if (hours_y(0.8*Nindiv) < hours_y(indiv) .and. hours_y(indiv) <= hours_y(1.0*Nindiv)) then
                qt_y(indiv) = 5.0D0                
                end if  
            end do
            
                        
            TR_H(:,time_y) = 0.0D0
            
            do indiv = 1, Nindiv     
             if (pqt_y(indiv)  == 1.0D0 .and. qt_y(indiv) == 1.0D0) then    
             temp_h(1,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 1.0D0 .and. qt_y(indiv) == 2.0D0) then    
             temp_h(2,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 1.0D0 .and. qt_y(indiv) == 3.0D0) then    
             temp_h(3,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 1.0D0 .and. qt_y(indiv) == 4.0D0) then    
             temp_h(4,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 1.0D0 .and. qt_y(indiv) == 5.0D0) then    
             temp_h(5,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 2.0D0 .and. qt_y(indiv) == 1.0D0) then    
             temp_h(6,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 2.0D0 .and. qt_y(indiv) == 2.0D0) then    
             temp_h(7,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 2.0D0 .and. qt_y(indiv) == 3.0D0) then    
             temp_h(8,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 2.0D0 .and. qt_y(indiv) == 4.0D0) then    
             temp_h(9,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 2.0D0 .and. qt_y(indiv) == 5.0D0) then    
             temp_h(10,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 3.0D0 .and. qt_y(indiv) == 1.0D0) then    
             temp_h(11,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 3.0D0 .and. qt_y(indiv) == 2.0D0) then    
             temp_h(12,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 3.0D0 .and. qt_y(indiv) == 3.0D0) then    
             temp_h(13,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 3.0D0 .and. qt_y(indiv) == 4.0D0) then    
             temp_h(14,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 3.0D0 .and. qt_y(indiv) == 5.0D0) then    
             temp_h(15,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 4.0D0 .and. qt_y(indiv) == 1.0D0) then    
             temp_h(16,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 4.0D0 .and. qt_y(indiv) == 2.0D0) then    
             temp_h(17,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 4.0D0 .and. qt_y(indiv) == 3.0D0) then    
             temp_h(18,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 4.0D0 .and. qt_y(indiv) == 4.0D0) then    
             temp_h(19,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 4.0D0 .and. qt_y(indiv) == 5.0D0) then    
             temp_h(20,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 5.0D0 .and. qt_y(indiv) == 1.0D0) then    
             temp_h(21,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 5.0D0 .and. qt_y(indiv) == 2.0D0) then    
             temp_h(22,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 5.0D0 .and. qt_y(indiv) == 3.0D0) then    
             temp_h(23,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 5.0D0 .and. qt_y(indiv) == 4.0D0) then    
             temp_h(24,indiv) = 1.000 
             elseif (pqt_y(indiv)  == 5.0D0 .and. qt_y(indiv) == 5.0D0) then    
             temp_h(25,indiv) = 1.000 
             end if
            end do
            
             TR_H(:,time_y) = sum(temp_h,2)/(0.2D0*Nindiv)            
             
             pqt_y = qt_y
                        
            end if
  
  end subroutine

  
  
  
subroutine dist1(INEQDIST, PDF, nq, qntle, ginic)


implicit none

    integer, intent(in) :: nq
    real(8), intent(in) :: INEQDIST(Nindiv), PDF(Nindiv)  
    real(8)  :: INEQCDF(Nindiv)    
    real(8), intent(out) :: qntle(nq), ginic
    real(8) :: mean
    integer :: row1, row2
    integer :: iq, jq, ii, jj
           
    
    
    do iq = 1, nq
        ii = ((iq-1.0D0)/nq)*Nindiv + 1
        jj = (1.0D0*iq/nq)*Nindiv
        qntle(iq)=sum(INEQDIST(ii:jj))/sum(INEQDIST)              
       
        !print *, ii, jj, qntle(iq)
        !pause
    end do
    
    
    
    
    ! Compute CDF
    INEQCDF(1) = INEQDIST(1)/sum(INEQDIST)        
                    
    do indiv = 2, Nindiv
    INEQCDF(indiv)  = INEQCDF(indiv-1) + INEQDIST(indiv)/sum(INEQDIST)            
    end do
    
        
    call gini(PDF,INEQCDF,Nindiv,ginic)
    
    qntle = qntle*100
      

end subroutine dist1

subroutine dist0(INEQDIST, PDF, nq, qntle, ginic)

! in_dist: distribution input
! in_pdf: pdf input
! n1 and n2 : size of in_dist
! nq = number of groups


! qntle: ouput distribution
! ginic: gini coefficients


implicit none

    integer, intent(in) ::  nq
    real(8), intent(in) :: INEQDIST(Nindiv), PDF(Nindiv)  
    real(8), allocatable, dimension(:) :: INEQCDF, INEQDIST2, PDF2
    real(8), intent(out) :: qntle(nq), ginic
    real(8) :: mean
    integer :: row1, row2, new_nu
    integer :: iq, jq, ii, jj
    
      
    

     
   jq = MINLOC(INEQDIST, dim=1, MASK = INEQDIST > 0.0D0)
   new_nu = Nindiv - jq +1
   
   !print *, iq, INEQDIST(iq-1), INEQDIST(iq),INEQDIST(iq+1)
   !pause
   
      
    
    allocate(INEQDIST2(new_nu), PDF2(new_nu), INEQCDF(new_nu))
    INEQDIST2= INEQDIST(jq:Nindiv)
    PDF2 = 1.0D0/new_nu 
    
   
    do iq = 1, nq
        
        ii = ((iq-1.0D0)/nq)*new_nu + 1
        jj = (1.0D0*iq/nq)*new_nu
        qntle(iq)=sum(INEQDIST2(ii:jj))/sum(INEQDIST2)        
        
        
    end do
    
    
    ! Compute CDF
    INEQCDF(1) = INEQDIST2(1)/sum(INEQDIST2)        
                    
    do indiv = 2, new_nu
    INEQCDF(indiv)  = INEQCDF(indiv-1) + INEQDIST2(indiv)/sum(INEQDIST2)            
    end do
    
        
    call gini(PDF2,INEQCDF, new_nu, ginic)
       
    
    qntle = qntle*100
    
     deallocate(INEQDIST2, PDF2,INEQCDF)
     


end subroutine dist0



subroutine gini(pop,cdf, nn, gg)

	implicit none
    
	integer::first, ig    
    integer, intent(in) ::nn
 	real(8), intent(in) :: pop(nn), cdf(nn)
	real(8), intent(out) ::gg
    real(8), allocatable :: s(:)
    
    allocate (s(nn))
    
    first = MINLOC(cdf,dim=1, MASK = cdf .GE. 0.0D0) 
      
    
    do ig = 1, first
        s(ig) = 0.0D0
    end do
        
    s(first) = pop(first)*cdf(first)*0.5
    
    do ig = first+1, nn
        s(ig) = pop(ig)*(cdf(ig) + cdf(ig-1))*0.5
    end do
               
	gg = (0.5 - sum(s))/0.5
    
    deallocate (s)

end subroutine gini
    

end module
