!	----------------------------------------------------------------------
!	File name: Tauchen.f90
!
!	This routine calculates the transition probability matrix of a Markov 
!	process. It is used to approximate the AR(1) process of shock. And the
!	calculated transition matrix is used for numerical integration when 
!	evaluating Emax functions.
!	----------------------------------------------------------------------	



module TauchenMod

    implicit none

	real(8):: normcdf
	external:: normcdf

contains

subroutine TauchenTr(mean, sigy, lambday, ygrid, prob)

	implicit none

	integer :: node, i, j
	real(8), intent(in) :: mean, sigy, lambday, ygrid(:)
	real(8), intent(out) :: prob(size(ygrid),size(ygrid))
	real(8) :: sig, ymax, ymin, ystep, cond_mean

	node = size(ygrid)
	
	sig = sigy*dsqrt(1.0d0-lambday**2.0d0)

	ymax = ygrid(node)
	ymin = ygrid(1)
	ystep = (ymax - ymin)/(node-1)

	!	calculate transition probability matrix

	do i = 1, node
		
		cond_mean = (1.0d0-lambday)*mean + lambday*ygrid(i)
		
		prob(i,1) = normcdf((ygrid(1) - cond_mean + 0.5d0*ystep)/sig)		&
				  - normcdf((ygrid(1) - cond_mean - 0.5d0*ystep)/sig)
		do j = 2, node-1
			prob(i,j) = normcdf((ygrid(j) - cond_mean + 0.5d0*ystep)/sig)	&
					  - normcdf((ygrid(j) - cond_mean - 0.5d0*ystep)/sig)
		end do
		prob(i,node) = normcdf((ygrid(node) - cond_mean + 0.5d0*ystep)/sig)	& 
					 - normcdf((ygrid(node) - cond_mean - 0.5d0*ystep)/sig)

		prob(i,:) = prob(i,:)/sum(prob(i,:))

	end do

end subroutine TauchenTr


subroutine TauchenNew(mean, sig, ygrid, prob)

	implicit none

	integer :: node, j
	real(8) :: mean, sig, ymax, ymin, ystep
	real(8), intent(in) :: ygrid(:)
	real(8), intent(out) :: prob(size(ygrid))

	node = size(ygrid)
	ymax = ygrid(node)
	ymin = ygrid(1)
	ystep = (ymax - ymin)/(node-1)

	!	calculate transition probability matrix
	
	prob(1) = normcdf((ygrid(1) - mean + 0.5d0*ystep)/sig)		&
			- normcdf((ygrid(1) - mean - 0.5d0*ystep)/sig)
	do j = 2, node
		prob(j) = normcdf((ygrid(j) - mean + 0.5d0*ystep)/sig) &
				- normcdf((ygrid(j) - mean - 0.5d0*ystep)/sig)
	end do
	prob(node) = normcdf((ygrid(node) - mean + 0.5d0*ystep)/sig)	&
			   - normcdf((ygrid(node) - mean - 0.5d0*ystep)/sig)

	prob = prob/sum(prob)

end subroutine TauchenNew

end module TauchenMod
