% A 272-sector model with the kinked demand curve
% WYP, 

close all

options = optimoptions('fsolve','FunctionTolerance',1e-8,'OptimalityTolerance',1e-8);
filenm  = 'kinked_linear_NS_homo_boundary';

%% defining parameters

parameters Pibar theta ppsi bet phipi phiy Ybar Lambar chi sigm rhoi;
parameters alph001 f001 PsP001 Lam001bar S001bar Z1001bar Z2001bar Z3001bar Z4001bar;

% calibrated parameters

Pibar   = (1.0434)^(1/4); % estimate of Carvalho, Nechio, Tristao (2021) Pre-Volcker
varphi  = 1;
bet     = 0.99;
theta   = 10;
chi     = 8;
sigm    = 1;
ppsi    = 1.1; % 1.0001

rhoi    = 0.8; % estimate of Carvalho, Nechio, Tristao (2021) rhoi1+rhoi2

phipi   = 0;
phiy    = 0;

J = 1;

%% load Nakamura-Setinsson frequencies
% d_regular | d_posted | weight (monthly duration)

NS_duration = load('duration_weight.txt');

d1 = NS_duration(:,1)/3;
d2 = NS_duration(:,2)/3;
weight = NS_duration(:,3);
weight = weight/sum(weight);

alpha1 = 1 - exp(-1./d1);
alpha2 = 1 - exp(-1./d2);

alpha1_mean = sum(weight.*alpha1);
alpha2_mean = sum(weight.*alpha2);

alphj   = alpha1_mean;
fj      = 1;

phipis  = 0:0.002:2.5;
phiyinc = 0.002; phiys  = (0:phiyinc:2);

phipin  = length(phipis);
phiyn   = length(phiys);

detchk  = nan(phipin,phiyn);
bnds    = cell(phipin,1);

%% solve for the steady state

% upper_Pibar = (1/bet/(1-min(alphs)))^(1/theta/2);
% if (Pibar > upper_Pibar)
%     break;
% end

ff = @(x) func_kinked_linear_NS_SS(x,J,bet,alphj,fj,Pibar,theta,sigm,chi,ppsi);
x0 = ones(J+1,1);
[x,fval,exitf] = fsolve(ff,x0,options);

if (exitf<1)
    error('SS not solved!')
end

Ybar          = exp(x(1));
PsPj          = exp(x(2:end));

Lam_j_powered = alphj./(1-(1-alphj)*Pibar^(theta*(1-ppsi)-1)).*(PsPj.^(1-theta*(1-ppsi)));
Lambarj       = Lam_j_powered.^(1/(1-theta*(1-ppsi)));

Lambar        = (1*Lam_j_powered)^(1/(1-theta*(1-ppsi)));

Sbarj         = alphj./(1-(1-alphj)/Pibar).*PsPj;

Z1barj        = Lambar^(theta*(1-ppsi))*Ybar^(1-sigm)./(1-bet*(1-alphj)*Pibar^(theta*(1-ppsi)-1));
Z2barj        = Ybar^(1-sigm)./(1-bet*(1-alphj)/Pibar);
Z3barj        = Lambar^(2*theta*(1-ppsi))*Ybar^2./(1-bet*(1-alphj)*Pibar^(2*theta*(1-ppsi)));
Z4barj        = Lambar^(theta*(1-ppsi))*Ybar^2./(1-bet*(1-alphj)*Pibar^(theta*(1-ppsi)));

% assign parameter values

alph001   = alphj(001);
f001      = fj(001);
PsP001    = PsPj(001);
Lam001bar = Lambarj(001);
S001bar   = Sbarj(001);
Z1001bar  = Z1barj(001);
Z2001bar  = Z2barj(001);
Z3001bar  = Z3barj(001);
Z4001bar  = Z4barj(001);

%% defining variables

var y i p lam;
var ps001 s001 lam001 z1001 z2001 z3001 z4001;

%% defining exogenous variables

varexo er;

%% shocks

shocks;
var er;
stderr 1;
end;

%% model equations in the same order as in our notes

model(linear);

y = y(+1) - 1/sigm*(i - p(+1));

lam = f001*(Lam001bar/Lambar)^(1-theta*(1-ppsi))*lam001;

lam001 = (1-alph001)*Pibar^(theta*(1-ppsi)-1)*(lam001(-1) - p) + (1-(1-alph001)*Pibar^(theta*(1-ppsi)-1))*ps001;

lam = ppsi*(f001*S001bar/Lambar*s001);

s001  = (1-alph001)/Pibar*(s001(-1) - p) + (1-(1-alph001)/Pibar)*ps001;

z1001 = (1-bet*(1-alph001)*Pibar^(theta*(1-ppsi)-1))*(theta*(1-ppsi)*lam + (1-sigm)*y) + bet*(1-alph001)*Pibar^(theta*(1-ppsi)-1)*(z1001(+1)+(theta*(1-ppsi)-1)*p(+1));
z2001 = (1-bet*(1-alph001)/Pibar)*(1-sigm)*y + bet*(1-alph001)/Pibar*(z2001(+1) - p(+1));
z3001 = (1-bet*(1-alph001)*Pibar^(2*theta*(1-ppsi)))*(2*theta*(1-ppsi)*lam + 2*y) + bet*(1-alph001)*Pibar^(2*theta*(1-ppsi))*(z3001(+1) + 2*theta*(1-ppsi)*p(+1));
z4001 = (1-bet*(1-alph001)*Pibar^(theta*(1-ppsi)))*(theta*(1-ppsi)*lam + 2*y) + bet*(1-alph001)*Pibar^(theta*(1-ppsi))*(z4001(+1) + theta*(1-ppsi)*p(+1));

(theta*(1-ppsi)-1)*Z1001bar*PsP001*(z1001+ps001) + ppsi*Z2001bar*PsP001^(theta*(1-ppsi)+1)*(z2001+(theta*(1-ppsi)+1)*ps001) = theta*chi*Z3001bar*PsP001^(-theta*(1-ppsi))*(z3001-theta*(1-ppsi)*ps001) - theta*chi*ppsi*Z4001bar*z4001;

i = rhoi*i(-1) + (1-rhoi)*(phipi*p + phiy*y) + er;

end;

% steady state

steady_state_model;

y=0; i=0; p=0; lam=0;
ps001=0; s001=0; lam001=0; z1001=0; z2001=0; z3001=0; z4001=0;

end;

%% loop

tic;

for ind6=1:phipin

    phipi = phipis(ind6);

disp('===============================================================')
fprintf('phipi = %8.4f (%i out of %i)\n',phipi,ind6,phipin)
disp('===============================================================')

    detchk0 = [];
    bnd0    = [];

for ind7 = 1:phiyn;

    phiy = phiys(ind7);

    check;

    deteig = sum(abs(oo_.dr.eigval)>1);
    detchk(ind6,ind7) = (deteig==M_.nsfwrd);

    if (~isempty(detchk0))
        if (detchk0~=detchk(ind6,ind7))
            bnd0 = [bnd0; phiy-0.5*phiyinc];
            detchk0 = detchk(ind6,ind7);
        end
    else
        detchk0 = detchk(ind6,ind7);
    end
        
end % ind7

    bnds{ind6} = bnd0;

end % ind6

toc;

save([filenm '.mat'],'phipis','phiys','detchk','bnds')
